///////////////////////////////////////////////////////////////////////////////  
//
//     File: Additive_Flute.c
// Platform: Single dsPIC Synth Board
//     Date: 2013-07-24
// Codename: Floot
//
// works as of 2013-08-05
//
// - verified that shadow attribute works with DAC1 ISR.
// - added ADSR in synth_components.c
// - implement velocity
// - replaced ADSR with assembly language
// - Added expression selectable waveform.  Velocity
//   controls how much raw sine and distorted sine is used
//   as well has how much 2nd harmonic is used.  More velocity
//   gives a less rounded and more robust sound.
// ver_e:
// - add noise "oscillator"
// ver_f:
// - lower sample rate and sound test - seems to be fine.
// ver_g:
// - Fix computations to give highest possible output.
//   This should improve the SNR.
// ver_h:
// - create a linear interpolation function (interp())
// - replaced velocity timbre control to sum of velocity and mod wheel
// - Added a single pole lowpass IIR to mod wheel to eliminate it's zipper
// - Add portamento - for now, time controlled by Y joystick
// ver_i:
// - Add pitchwheel support
// ver_j:
// - With only 60 clocks to work with and no ADC support, reducing SR to
//   56.xx kHz gives us enough to work with.  However, NCO1 can have a strong
//   6th harmonic, so we will change NCO1 not to select a wave form, rather
//   it will select between silence and the 2nd harmonic only.
// ver_k
// - Add ADC1 ports
//     ADC0:  noise
//     ADC1:  portamento time
// ver_l
// - Fixed "adsr" issue (which was really not the ADSR's fault)
// - ADC0 controls noise
// - ADC1 controls portamento time
// - cleaned up code, removed need for two slider values in interp() function
// - added a union for pitchwheel to eliminate some shifting
//
///////////////////////////////////////////////////////////////////////////////  

/*
;------------------------------------------------------------------------------
;
; ANALOG PINS:
;   ACTUAL PIN NUMBER    AN
;   2                    AN0
;   3                    AN1
;   6                    AN4
;   7                    AN5
;
;------------------------------------------------------------------------------
*/

#include <stdio.h>
#include "p33FJ128GP802.h"
#include "init.h"
#include "synth_components.h"
#include "MIDI_controller.h"

#define MIDI_LED 1

void delay( int );

// FUSES:
_FGS( GSS_OFF & GCP_OFF & GWRP_OFF );                            // Code Protection
_FOSCSEL( FNOSC_PRIPLL & IESO_ON );                              // Oscillator (20MHz ext Xtal) with 2 speed start-up
_FOSC( FCKSM_CSDCMD & IOL1WAY_OFF & OSCIOFNC_OFF & POSCMD_HS );  // Oscillator (20MHz ext Xtal) with 2 speed start-up
_FWDT( FWDTEN_OFF & WINDIS_OFF );                                // Watchdog
_FPOR( ALTI2C_ON & FPWRT_PWR16 );                                // Power up options (16ms)
_FICD( ICS_PGD1 & JTAGEN_OFF );
//////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////

// GLOBAL STORAGE
int LeftSamp = 0;
int RightSamp = 0;

// these locations are defined in an assembly language module, but the declaration also needs to be here
extern int AN0_VAL;
extern int AN1_VAL;
extern int AN4_VAL;
extern int AN5_VAL;

int ADC0, ADC1, ADC2, ADC3;

int GATES = 0;
int NOTE0 = 0;
int VEL0 = 0;
int MOD_WHEEL = 0;
int JOYSTICK_Y = 127;
int JOYSTICK_X = 0;
int PITCH_WHEEL = 0x4000;            // PITCH_WHEEL is delivered as fixed point
unsigned long PHINClo = 0;           // low reading from tuning table
unsigned long PHINChi = 0;           // high reading from tuning table

unsigned long phase_accum0 = 0;
unsigned long PHINC0 = 0;
unsigned angle0 = 0;

unsigned long phase_accum1 = 0;
unsigned long PHINC1 = 0;
unsigned angle1 = 0;

unsigned long PHINC0p = 0;           // PHINC0 with portamento applied

unsigned ADSR0state = 0;
int ADSR0value = 0;

unsigned ADSR1state = 0;
int ADSR1value = 0;

unsigned ADSR2state = 0;
int ADSR2value = 0;

int Samp0 = 0;
int Samp1 = 0;

int vel = 0;
int timbre = 0;
int interp0 = 0;
int interp1 = 0;
int filt_vel = 0;

int mod = 0;
int filt_mod;

int sin_val;
int dist_sin_val;

int tmp0, tmp1, tmp2;
long tmp0L, tmp1L;

int noiz = 0;
int filtered_noiz;
int nz;

long port_speed = 0;

long PW_N = 0x00000000L;

union                            // union with struct for storing and
  {                              // manipulating pitchwheel without
  long PWlong;                   // the need for shifting.
  struct
    {
    unsigned int PWlo;           // low word
    int PWhi;                    // believe it or not, this is where the high word is.
    } s;                         // struct is named 's'
  } PW;                          // union is named PW

int ADC0, ADC1, ADC2, ADC3;

extern void delay( int );

int NOTE0a = 0;

int main ( void )
{   
// HIGH SPEED CRYSTAL CLOCK:
CLKDIV = 0x0002;
PLLFBD = 0x001E;                // M = PLLFBD + 2
__builtin_write_OSCCONH(0x03);  // New oscillator Primary oscillator (XT, HS, EC) with PLL
__builtin_write_OSCCONL(0x01);  // Enable clock switch as per dsPIC oscillator start-up
while ( OSCCONbits.LOCK != 1 ); // Wait for PLL to lock

////////////////////////////////////////////////////////////
//// Peripheral Modules Initializations

init_DAC1(9);    // 6  = 89.285 khz
                 // 7  = 78.125 khz
                 // 8  = 69.444 khz
                 // 9  = 62.500 khz
                 // 10 = 56.818 khz
                 // 11 = 52.083 khz
                 // 12 = 48.076 khz
init_ADC1();     // initialize ADC and DMA channel 0
init_UART1();
init_UART2();    // MIDI UART
init_LED();

////////////////////////////////////////////////////////////
/// Program initializations (these are routines that are to be done ONCE at power up)

seed( 0x630FE891L );   // seed the random number generator
PW.PWlong = 0L;        // initialize pitchwheel union
MIDI_Controller();     // a function that never returns.

return(0);       // This needs to be here or a "control reaches end of non-void function" warning is issued.
}

/////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////            /////////////////////////////////////////////////////////
////////////////////////////////////////////////  DAC1 ISR  /////////////////////////////////////////////////////////
////////////////////////////////////////////////            /////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

void __attribute__((interrupt, no_auto_psv, shadow)) _DAC1RInterrupt(void)
{
asm("PUSH W8");
asm("PUSH W9");
asm("PUSH W10");
asm("PUSH W11");
asm("PUSH W12");
asm("PUSH W13");
asm("PUSH W14");
IFS4bits.DAC1RIF = 0;		  // Clear the DAC1R Interrupt Flag;

DAC1LDAT = LeftSamp;    // transfer left sample to left DAC data register
DAC1RDAT = RightSamp;   // transfer right sample to right DAC data register

IFS0bits.AD1IF = 0;  // Clear the AD1IF bit   (Interrupt Flag)

// Get ADC values already acquired by ADC hardware
ADC0 = (AN0_VAL<<3);   // convert to fixed point
ADC1 = (AN1_VAL>>5);   // convert to 7 bit (unsigned 8 bit)
//ADC2 = AN4_VAL;
//ADC3 = AN5_VAL;

DMA0CONbits.CHEN = 1;            // start conversions for next cycle (CHEN is 15)

////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////
//               DO NOT EDIT ABOVE THIS LINE 
////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////
// YOUR SYNTH CODE GOES HERE:
// YOUR SYNTH CODE GOES HERE:
// YOUR SYNTH CODE GOES HERE:

// PHINClo and PHINChi are values fetched from a phase increment table (the tuning table).  The address of that table is based
// on note numbers.  Each fetch from the table returns 2 24 bit phase increment values.  One value is 1/2 step lower than the note 
// number.  The other value is one half step higher.  With the pitch wheel in the neutral position this will select the note in 
// between.  The pitch wheel in other positions selects a phase increment that will be the interpolation between the two table reads 
// selected by the pitch wheel fixed point value.  This paradigm sets pitchwheel action to 1/2 step above and below the note played.
//
// PHINClo is the lower value, PHINChi is the higher value

PW.s.PWhi = PITCH_WHEEL;
PW_N =  0x7FFFFFFFL - PW.PWlong;                               // create PW_N from PW_P
//PW_N =  (long)( 0x80000000UL - (unsigned long)PW.PWlong );    // create PW_N from PW_P

tmp0L = widemul( PW.PWlong, PHINChi );                         // proportion of the low table value
tmp1L = widemul( PW_N, PHINClo );                              // proportion of the high table value
PHINC0 = tmp0L + tmp1L;                                        // add them to make a phase increment selected by the pitch wheel.

portamento( &PHINC0, &PHINC0p, ADC1, 50 );                     // Do portamento on PHINC0 to PHINC0p at rate JOYSTICK_Y with skip sample factor 50 

PHINC1 = PHINC0p << 1;                                         // PHINC1 must be an octave about PHINC0.
vel = VEL0 << 7;                                               // vel is the fixed point version of VEL0

lowpass1pole( vel, &filt_vel, 500 );                           // Filter velocity changes to eliminate transients.

mod = (MOD_WHEEL << 7);                                        // mod is fixed point version of MOD_WHEEL
lowpass1pole( mod, &filt_mod, 250 );                           // filtering removes zipper noise

timbre = filt_vel + filt_mod;                                  // timber is a selector being the sum of velocity and mod wheel.

///////////////////////////////////////////////////////
// NCO_0   Fundamental and Distorted Fundamental
//         The amount of distorted signal 
//         is selected by the variable timbre
angle0 = saw24( &phase_accum0, &PHINC0p );                     // generate a sawtooth waveform with a 24 bit phase accumulator.  angle0 is a fixed point angle.
ADSR( &ADSR0state, &ADSR0value, 20, 9, 23000, 3 );             // do one ADSR0 tick
sin_val      = sine( angle0 );                                 // sine lookup
dist_sin_val = dist_sine( angle0 );                            // distorted sine lookup

// waveform interpolator
interp0 = interp( dist_sin_val, sin_val, &timbre );            // interpolate from sin_val to dist_sin_val using timbre as a selector
fastmul( ADSR0value, interp0, &Samp0 );                        // apply ADSR0 output to control amplitude of the interpolation output (waveform select).

///////////////////////////////////////////////////////
// NCO_1   Second Harmonic
//         The amount of second harmonic
//         is selected by the variable timbre
angle1 = saw24( &phase_accum1, &PHINC1 );                      // generate sawtooth waveform with 24 bit phase accumulator.  angle1 is fixed point angle.
ADSR( &ADSR1state, &ADSR1value, 40, 9, 32000, 3 );             // do one ADSR1 tick

sin_val = sine( angle1 );                                      // sine lookup
fastmul( timbre, sin_val, &interp1 );                          // scale interp1 as timbre * sin_val  -  interp1 is the scaled second harmonic
fastmul( ADSR1value, interp1, &Samp1 );                        // scale Samp1 as ADSR1 output and interp1 (scaled second harmonic)

//////////////////////////////////////////////////////
// blow noise.
noise( &noiz );                                                // Shift the LFSR once and grab a fixed point noise value

fastmul( noiz, ADC0, &tmp1 );                                  // ADC0 controls how much noise

if ( NOTE0 > 35 && NOTE0 < 96 ) NOTE0a = NOTE0;
lowpass1pole( tmp1, &filtered_noiz, ((NOTE0a-35)<<4) );        // lowpass filter the noise, use a lower Fc for lower note numbers.

ADSR( &ADSR2state, &ADSR2value, 300, 10, ((ADC0>>2)), 3 );     // do one ADSR2 tick

fastmul( ADSR2value, filtered_noiz, &tmp2 );                   // apply ADSR2 envelope to the filtered noise signal (to tmp2)
fastmul( tmp2, filt_vel, &tmp2);                               // scale enveloped noise by filt_vel (velocity)

///////////////////////////////////////////////////////
// NCOs Waveforms Mixer
tmp0 = interp( (Samp1>>1), (Samp0>>1), &filt_vel );            // Velocity selects the waveform here
LeftSamp = RightSamp = ( tmp0 + tmp2 );                        // Mix the two waveforms and noise (tmp2) to both channels

//delay(54);   // with portamento
//delay(50);   // with mod wheel filter
//delay(20);   // with pitch wheel support
//delay(26);    // with single sine wave on NCO1

// YOUR SYNTH CODE GOES HERE:
// YOUR SYNTH CODE GOES HERE:
// YOUR SYNTH CODE GOES HERE:
////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////
//               DO NOT EDIT BELOW THIS LINE 
////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////

// The interrupt flag should still be clear if the ISR completes before the next interrupt comes in.
// If the interrupt flag is on here, it means that this ISR is too slow.


// make sure to turn MIDI LED off in MIDI controller
#ifdef TEST_LED
if ( IFS4bits.DAC1RIF == 1 ) asm( "BSET LATA, #4" );
else                         asm( "BCLR LATA, #4" );
#endif

asm("POP W14");
asm("POP W13");
asm("POP W12");
asm("POP W11");
asm("POP W10");
asm("POP W9");
asm("POP W8");
}
