;
;     Project: MIDI_Controller.s
;    Engineer: Scott Gravenhorst
;        Date: 2013-07-27
;
; Description: MIDI controller for monosynths
;
;     Version: 0.0.1   - First cut
;                        In this version, the MIDI UART is polled once for each
;                        DAC interrupt inside the DAC DMA ISR.
;                      - We may need to move the storage declaration to the C file
;                        (which contains the main() function, device init and ISRs.
;                      - All timing is based upon the DAC DMA ISR.
;                      - This code will support up to 8 voices as it seems a bit
;                        hopeful that a single dsPIC would support more.  Likely the
;                        actual number of voices will be 4 or less.
;                      - SPI controller is removed.
;                      - MIDI UART setup is removed.
;                      - MIDI UART ISR is removed.
;                      - Legato is implemented
;
;------------------------------------------------------------------------------
; System Exclusive Message structure:
; MFR ID
; MODEL NUMBER
; UNIT NUMBER
; PARAMETER ADDRESS MSB
; PARAMETER ADDRESS LSB
; PARAMETER DATA 
;------------------------------------------------------------------------------
;
; This program is designed to work in concert with a monosynth running as part 
; of the DAC ISR.  Note on messages will set _GATES bit zero and cause this MIDI 
; controller code to fetch a 24 bit phase increment value from the tuning table 
; and store it in location _PHINC0 which is defined in the calling C program.
; Note off messages make no change to PHINC0, but zero _GATES.
;
; NOTE0, VEL0, GATES and MOD_WHEEL are all declared globally in C.

;


.include "p33FJ128GP802.inc"
		
.text

;------------------------------------------------------------------------------
; Application Equates
;------------------------------------------------------------------------------
  
 .equ MFR_ID,       0x7F      ;  CONSTANT MFR_ID,               7F           ; my mfr ID - Change this if you want a different ID
 .equ MODEL_NUMBER, 0x7F      ;  CONSTANT MODEL_NUMBER,         7F           ; 00 = GateMan-I; 01 = PolyDaWG; 02 = GateManPoly
 .equ UNIT_NUMBER,  0x00      ;  CONSTANT UNIT_NUMBER,          00           ; unit number

 .equ MIDI_TRANSPOSE, 0x14    ;  CONSTANT MIDI_TRANSPOSE,            14      ; how much to SUBTRACT from MIDI note numbers for note on/off messages
                                                                             ; this gives more range on the high end
; MIDI CONSTANTS
; message type constants
 .equ K_NoteOff,            0x80  ; CONSTANT K_NoteOff,            80
 .equ K_NoteOn,             0x90  ; CONSTANT K_NoteOn,             90
 .equ K_PolyKeyPressure,    0xA0  ; CONSTANT K_PolyKeyPressure,    A0
 .equ K_ControllerChange,   0xB0  ; CONSTANT K_ControllerChange,   B0
 .equ K_ProgramChange,      0xC0  ; CONSTANT K_ProgramChange,      C0
 .equ K_ChannelPressure,    0xD0  ; CONSTANT K_ChannelPressure,    D0
 .equ K_PitchBend,          0xE0  ; CONSTANT K_PitchBend,          E0
 .equ K_System,             0xF0  ; CONSTANT K_System,             F0

; Controller Change byte 1 constants:
 .equ MOD_WHEEL,            0x01  ; CONSTANT MOD_WHEEL,            01
 .equ SUSTAIN,              0x40  ; CONSTANT SUSTAIN,              40
 .equ JOYSTICK_X,           0x10  ; CONSTANT JOYSTICK_X,           10           ; Korg Wavestation, don't know about others.
 .equ JOYSTICK_Y,           0x11  ; CONSTANT JOYSTICK_Y,           11           ; Korg Wavestation, don't know about others.
 .equ ALL_NOTES_OFF,        0x7B  ; CONSTANT ALL_NOTES_OFF,        7B

 ;------------------------------------------------------------------------------
 ;Sec 0.3.Global Declarations
 ;------------------------------------------------------------------------------
 .global _MIDI_Controller
 .global _PHINC0

 .global _AN0_VAL
 .global _AN1_VAL
 .global _AN4_VAL
 .global _AN5_VAL


 ;------------------------------------------------------------------------------
 ;Sec 0.4.Global Initialised Variables
 ;------------------------------------------------------------------------------
 
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;                  ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;; SCRATCH PAD RAM  ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;                  ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

 .section .data
;-------------------------------------
; WORD STORAGE
;------------------------------------- 


  MIDI_LED_COUNTER:       .word 0x0000


  ROUNDROBIN:             .word 0x0000

  SYNTH_GATE:             .word 0x0000               ; Used by SendGATE code segment and UPDATE_GATE subroutine
  SYNTH_SUSTAIN:          .word 0x0000
  ;SYNTH_JOYSTICK_X:       .word 0x0000
  ;SYNTH_JOYSTICK_Y:       .word 0x0000
  SYNTH_GATES:            .word 0x0000
  SYNTH_CHANNEL_PRESSURE: .word 0x0000
  ;SYNTH_PITCH_WHEEL_MSB:  .word 0x0000
  ;SYNTH_PITCH_WHEEL_LSB:  .word 0x0000

; DATAbyteCOUNT           Is the counter that controls what happens to MIDI data bytes.
; DATAbyteCOUNTrunstat    Holds the data byte count value to load into DATAbyteCOUNT to refresh
;                         it when a message comes in assuming valid running status.
  DATAbyteCOUNT:          .word 0x0000
  DATAbyteCOUNTrunstat:   .word 0x0000

  DATAbyte1:              .word 0x0000               ; for current message storage of data byte 1
  DATAbyte2:              .word 0x0000               ; for current message storage of data byte 2

; These variables will be defined in the C program

;  _NOTE0: .word 0x0000
;  _NOTE1: .word 0x0000
;  _NOTE2: .word 0x0000
;  _NOTE3: .word 0x0000
;  _NOTE4: .word 0x0000
;  _NOTE5: .word 0x0000
;  _NOTE6: .word 0x0000
;  _NOTE7: .word 0x0000

;  _VEL0: .word 0x0000
;  _VEL1: .word 0x0000
;  _VEL2: .word 0x0000
;  _VEL3: .word 0x0000
;  _VEL4: .word 0x0000
;  _VEL5: .word 0x0000
;  _VEL6: .word 0x0000
;  _VEL7: .word 0x0000


; This is the legato keys down table:

  KeyDownCount:    .word 0x0000           ;  key down counter for legato
  KDptr:           .word 0x0000
  KD00:            .word 0x0000
  KD01:            .word 0x0000
  KD02:            .word 0x0000
  KD03:            .word 0x0000
  KD04:            .word 0x0000
  KD05:            .word 0x0000
  KD06:            .word 0x0000
  KD07:            .word 0x0000
  KD08:            .word 0x0000
  KD09:            .word 0x0000
  KD10:            .word 0x0000
  KD11:            .word 0x0000
  KD12:            .word 0x0000
  KD13:            .word 0x0000
  KD14:            .word 0x0000
  KD15:            .word 0x0000

  SUS:             .word 0x0000          ; used to save sustain pedal state.

  RUNNINGstatus:   .word 0x0000          ; holds the current MIDI running status value
  MessageTYPE:     .word 0x0000          ; Message type, i.e., high nybble
  MessageCHANNEL:  .word 0x0000          ; Message channel, i.e., low nybble
  COMMANDbyte:     .word 0x0000          ; command byte, contains command nybble and channel nybble of current message
  MIDI_CHAN:       .word 0x0000          ; for now, this is just channel 1, we need to figure out a way to change this value
  TRANSPOSE:       .word 0x0000          ; a potentially user parameter
  SYSEX:           .word 0x0000          ; sysex state machine state register
  MIDIinputBYTE:   .word 0x0000          ; storage for MIDI data written by ISR
  MESSAGEcomplete: .word 0x0000
  LED_COUNTER:     .word 0x0000


;-------------------------------------
; BYTE STORAGE
;------------------------------------- 

;  SYSEX_MSB:            .byte 0x00
;  SYSEX_LSB:            .byte 0x00




;------------------------------------------------------------------------------

;------------------------------------------------------------------------------

; DMA buffer allocation and naming.  Buffers are declared at the end of SRAM

;------------------------------------------------------------------------------

; Note below "align(8)" ensures that the block of 8 bytes defined below will

; start on an 8 byte boundary.


;------------------------------------------------------------------------------
;Sec 1.0.Main Code
;------------------------------------------------------------------------------
 
 .section .text

 _MIDI_Controller:

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;; MIDI CONTROLLER CODE STARTS HERE ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;


  CLR MESSAGEcomplete
  CLR MIDI_CHAN
  CLR TRANSPOSE
  CLR _GATES
  CLR SYSEX
  CLR ROUNDROBIN
  INC ROUNDROBIN


  CLR KeyDownCount
  MOV #KD00, W0

  MOV W0, KDptr


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;  
; Wait for and process MIDI Bytes.
; This is the main dispatch loop that monitors the ISR's regMIDIbyteVALID flag register and MESSAGE_COMPLETE flag.

idle:
;  BTST.B _MIDI_byte_available, #0

  BTST U2STA, #URXDA                         ; test data available bit
  BRA NZ, GetMIDIbyte                        ;


  BTST MESSAGEcomplete, #0                   ;
  BRA NZ, ProcessMessage                     ;



  CP0 MIDI_LED_COUNTER

  BRA NZ, MIDI_LED_timer

  BCLR LATA, #4

  BRA idle

MIDI_LED_timer:

  DEC MIDI_LED_COUNTER

  GOTO idle                                  ;
  
; this is some system exclusive code that is turned off

;  MOV #0xF7, W0
;  CP W0, W14                                ; w14 is MIDIbyte
;  BRA NZ, CheckForF0                        ;
;  CLR SYSEX                                 ; set state machine to idle
;  GOTO idle                                 ;

; System Exclusive message handling is disabled, it is not needed in this synth (yet)
;CheckForF0:
;  MOV #0xF0, W0                             ; is this byte the start of a new sysex message?
;  CP W0, W14                                ;
;  BRA NZ, SysExStateMachine                 ; state machine will interpret the byte
;  MOV #0x01, W0                             ; start the sysex state machine
;  MOV W0, SYSEX                             ; Set the sysex state machine state to 01 (start)
;  GOTO idle                                 ; no need to process further

GetMIDIbyte:
  MOV U2RXREG, W14                           ; this action also clears U2RXDA



; Throw away active sensing bytes

  MOV #0x00FE, W0

  CP W0, W14

  BRA Z, idle                                ; if it's active sensing, then just ignore

  MOV #0x3FFF, W0                            ; move LED timer value 

  MOV W0, MIDI_LED_COUNTER                   ; to the counter

;-----------------------

;; MIDI LED
;; MIDI LED
;; MIDI LED
  BSET LATA, #4

;; MIDI LED
;; MIDI LED
;; MIDI LED
;-----------------------


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; Procbyte or "Process byte" is an entry point that is JUMPed to for MIDI performance data input activity

Procbyte:
; there is a byte to process

  BTST.Z W14, #7                            ; is high bit set?
  BRA NZ, MIDI_STATUS_BYTE                  ;

; We have a data byte because bit 7 is zero.
; check if we're in sysex mode:
  CP0 SYSEX                                 ; if SYSEX == 0 then not in sysex mode
  BRA Z, PerformanceData                    ;

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;
; SYSTEM EXCLUSIVE MESSAGES
;
; Sysex is handled with a state machine.  Register SYSEX holds the current state.
; Setting the state to 7F causes the state machine to ignore the rest of any sysex message
; and wait for a F7 byte.
;
  GOTO idle ; just in case.  system exclusive processing is disabled in this version.
;SysExStateMachine:
;  MOV SYSEX, W1
; 
;  MOV #1, W0
;  CP W1, W0                      ;
;  BRA Z, CheckID                 ; check mfr ID
;  
;  MOV #2, W0
;  CP W1, W0                      ;
;  BRA Z, ModelNumber             ; check model number
;
;  MOV #3, W0
;  CP W1, W0                      ;
;  BRA Z, UnitNumber              ; check unit number
;
;  MOV #4, W0
;  CP W1, W0                      ;
;  BRA Z, ParameterAddrMSB        ; set parameter address high 7 bits
;
;  MOV #5, W0
;  CP W1, W0                      ;
;  BRA Z, ParameterAddrLSB        ; set parameter address low 7 bits
;
;  MOV #6, W0
;  CP W1, W0                      ;
;  BRA Z, ParameterData           ; set parameter data
;  CP W1, W0                      ;
;
;  GOTO idle                      ;
;
;CheckID:
;  MOV #0x02, W0                  ; move to next state, check UnitNumber
;  MOV WREG, SYSEX
;
;  MOV #MFR_ID, W0
;  CP W0, W14                     ; this device's mfr ID
;  BRA Z, idle                    ; if this is ours, go to idle and wait
;  MOV #0x7F, W0                  ; not our message, bypass rest of message.
;  MOV WREG, SYSEX
;  GOTO idle                      ;
;
;ModelNumber:
;  MOV #3, W0                     ;
;  MOV WREG, SYSEX
;
;  MOV #MODEL_NUMBER, W0
;  CP W0, W14                     ; Compare with our model number
;  BRA Z, idle                    ; if this is ours, go to idle and wait
;  MOV #0x7F, W0                  ; not our message, bypass rest of message.
;  MOV WREG, SYSEX
;  GOTO idle                      ;
;
;UnitNumber:
;  MOV #4, W0                     ; set for parameter addr MSB mode
;  MOV WREG, SYSEX
;
;  MOV #UNIT_NUMBER, W0
;  CP W0, W14                     ; compare message type byte, 00=parameter
;  BRA Z, idle                    ;
;  MOV #0x7F, W0                  ; not our message, bypass rest of message.
;  MOV WREG, SYSEX
;  GOTO idle                      ;
;  
;ParameterAddrMSB:
;  MOV #5, W0                     ; set for parameter addr LSB mode
;  MOV WREG, SYSEX
;  MOV W14, W0                    ; get the MSB byte
;  MOV.B WREG, SYSEX_MSB          ; store the byte in the external parameter selection hardware
;  GOTO idle                      ;
;  
;ParameterAddrLSB:
;  MOV #6, W0                     ; set for parameter data mode
;  MOV WREG, SYSEX                ;
;  CLR W12                        ; save the address byte in W12
;  MOV W14, W12                   ; we use W12 as sE, a pointer
;  GOTO idle                      ;
;  
;ParameterData:
;  MOV #0x7F, W0                  ; set to wait for F7
;  MOV WREG, SYSEX
;
; These instructions will remain unimplemented until we design a SYSEX structure for the dsPIC
;  OUTPUT regMIDIbyte, (sE)       ; send data byte to port number in sE (the address byte).

;  LOAD sE, 00
;  OUTPUT sE, SYSEX_MSB_PORT      ; after sending parameter byte, set sysex MSB port to zero to disable any other transfers
  
;  GOTO idle                      ;
  
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

PerformanceData:
  DEC DATAbyteCOUNT                          ; decrement the byte counter...
  BRA Z, byte2                               ; if equal to 0 now, it was 1, so this is byte2 ... numbers are fun.

  MOV W14, DATAbyte1                         ; save this byte as data byte 1
  GOTO idle                                  ;

byte2:                                       ; in all cases, this is the last data byte.  Save it and set MESSAGE_COMPLETE
  MOV W14, DATAbyte2                         ; save this byte as data byte 2

  BSET MESSAGEcomplete, #0                   ; set message complete bit

  MOV DATAbyteCOUNTrunstat, W0
  MOV W0, DATAbyteCOUNT                      ; set the count again, next message could be running status data.

  GOTO idle                                  ;
  
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; This is a status or command byte.

MIDI_STATUS_BYTE:                          ; high bit is set, now determine what sort of message this is
  MOV W14, W0                              ; W14 has the last MIDI byte received
  MOV #0xF0, W1                            ; 0xF0 is AND mask
  AND W0, W1, W0                           ; AND the MIDI byte with mask to get the message type 
  MOV W0, MessageTYPE                      ;
  
  MOV W14, W0                              ;
  MOV #0x0F, W1                            ; AND mask for MIDI channel
  AND W0, W1, W0                           ; AND to get the MIDI channel to W0
  MOV W0, MessageCHANNEL                   ;

  MOV MessageTYPE, W0                      ; 
  MOV #K_System, W1                        ;
  CP W0, W1                                ; Is it a system message? ( 0xF0 )
  BRA NZ, NotFx                            ;
  
; We are here because this byte if Fx
; decode and act on other Fx messages here.  For now, it's just a jump to idle
  MOV #0xF7, W0                            ; test for end of SysEx message
  CP W0, W14                               ; end of sysex message
  BRA NZ, NotF7                            ;
  CLR SYSEX                                ; sysex off.
  GOTO idle                                ;
  
NotF7:
  MOV #0xF0, W0                            ; test for start of SysEx message
  CP W14, W0                               ; F0 ?  (start of sysex message)
  BRA NZ, idle                             ;
;; We have an F0
  MOV #1, W0                               ; set sysex state flag to check mfr ID
  MOV WREG, SYSEX
  GOTO idle                                ;
  
NotFx:
  CLR SYSEX                                ; sysex always goes off on any status byte.
  MOV W14, COMMANDbyte                     ; remember this message's command byte.

  MOV #K_ProgramChange, W0                 
  MOV MessageTYPE, W1
  CP W0, W1                                ; is this PROGRAM CHANGE ?
  BRA Z, ONE_DATA_BYTE                     ;
  
  MOV #K_ChannelPressure, W0
  CP W0, W1                                ; is this CHANNEL PRESSURE ?
  BRA Z, ONE_DATA_BYTE                     ;

  MOV #2, W0                               ; This message has 2 databytes
  MOV W0, DATAbyteCOUNT                    ; If not channel pressure or program change, then 2 bytes of data
  GOTO SetCount                            ;


ONE_DATA_BYTE:
  MOV #1, W0                               ;
  MOV W0, DATAbyteCOUNT                    ;

; set count source for replenishing DATAbyteCOUNT when no status byte is sent (running status)
SetCount:
  MOV DATAbyteCOUNT, W0                    ;
  MOV W0, DATAbyteCOUNTrunstat             ;

  GOTO idle                                ;

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;  
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;                   ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;  PROCESS MESSAGE  ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;                   ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

; A complete message is ready to process...
; Message type determines the number of data bytes.

ProcessMessage:
  BCLR MESSAGEcomplete, #0                   ; clear the message complete flag

; See if this is our message.  If not, go idle, if yes, do the message.
  MOV COMMANDbyte, W0                        ; get the current command byte into TEMP
  MOV #0x0F, W2
  AND W2, W0, W0                             ; isolate only the channel.
  MOV MIDI_CHAN, W1                          ; get current MIDI channel number for this hardware

  CP W0, W1                                  ; compare MIDI channel setting with current running status channel number
  BRA NZ, idle                               ; JUMP to idle if not our channel number.

  MOV COMMANDbyte, W0  
  MOV W0, RUNNINGstatus                      ; this is a message on my channel so set running status.

  MOV RUNNINGstatus, W0
  MOV W0, MessageTYPE                        ;
  MOV #0xF0, W1
  AND W1, W0, W0
  MOV W0, MessageTYPE                        ; get message type nybble

; This is where we act on complete messages.
; JUMP table of vectors to different message type needs.
; Note that W0 already contains MessageTYPE

  MOV #K_NoteOff, W1
  CP W0, W1                                  ;
  BRA Z, NOTE_OFF                            ;

  MOV #K_NoteOn, W1
  CP W0, W1                                  ;
  BRA Z, NOTE_ON                             ;

  MOV #K_PolyKeyPressure, W1
  CP W0, W1                                  ;
  BRA Z, POLY_KEY_PRESSURE                   ;

  MOV #K_ControllerChange, W1
  CP W0, W1                                  ;
  BRA Z, CONTROLLER_CHANGE                   ;

  MOV #K_ProgramChange, W1
  CP W0, W1                                  ;
  BRA Z, PROGRAM_CHANGE                      ;

  MOV #K_ChannelPressure, W1
  CP W0, W1                                  ;
  BRA Z, CHANNEL_PRESSURE                    ;

  MOV #K_PitchBend, W1
  CP W0, W1                                  ;
  BRA Z, PITCH_BEND                          ;

  GOTO idle                                  ; anything else, toss.
  
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; This section will:
;; Process note on and note off messages.
;; for note on, toggle the gate signal if it's on,  turn gate on if it's off.
;;

; for dsPIC, we will use W8 for s8 (TRANSPOSE). 
; TRANSPOSE is no longer a port, it's an initialized value in RAM
NOTE_ON:
  MOV TRANSPOSE, W8
  CP0 DATAbyte2                              ; test if velocity byte is zero (alternative NOTE_OFF)
  BRA NZ, PLUCK                              ; if not, it's a real note-on

NOTE_OFF:                                    ; we are here either because of a real note-off or 

  MOV TRANSPOSE, W8                          ; a note on with velocity zero.
; This is a NOTE OFF message, whether it's a NOTE ON with zero vel or a real note off.
;
; Manage the key down table
  DEC KeyDownCount                           ; decrement
; find the key in the table
  MOV DATAbyte1, W2                          ; get the note off key number
  MOV #KD00, W1                              ; for table compress
  MOV #KD15, W3                              ; for table compress
search:  
  CP W2, [W1++]                              ; compare note value to table value
  BRA NZ, search                             ; if not zero, then the table value is not this note number, search again

; W1 is pointing to the next entry after the found one.
move:
  MOV [W1], [--W1]                           ; Move the cell after the one to delete into the cell to delete
  ADD W1, #4, W1                             ; Add 4 to W1 to put W1 pointing to the next value to move
  CP W1, W3                                  ; compare W1 with the end of the table.
  BRA LE, move                               ; if <= then we can do another move.

; when here, the table has been compressed
  DEC2 KDptr                                 ; decrement the new data table pointer since the table is one location smaller.
  MOV KDptr, W1
  MOV [--W1], W0
  MOV W0, _NOTE0                             ; supply the end table value as the new note number

  CP0 KeyDownCount
  BRA NZ, ChangePitch                        ; at least one key is down

  CLR W9
  MOV W9, _GATES                             ; save the new state
  GOTO idle

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
PLUCK:
; We get here when we have received a real NOTE ON message.

; manage key down table
  MOV DATAbyte1, W0                  ; a new note on
  MOV KDptr, W1                      ; get key down table pointer
  MOV W0, [W1++]                     ; store note number into key down table
  MOV W1, KDptr                      ; store incremented pointer
  INC KeyDownCount                   ; increment the key down counter

  MOV SUS, W0                        ; Check sustain pedal state
  CP0 W0                             ; note: FETCH instruction does not change flags
  BRA Z, PEDALUP                     ;

; The pedal is held down.  So first try to use an inactive voice before stealing
; First look for an unused voice by checking for unbusy voices.
; This needs to take into account that some keys can be held down
; and should not be stolen if there are any unbusy voices.

PEDALUP:
  MOV _GATES, W0
  MOV #0xFFFF, W1
  CP W0, W1
  BRA Z, RR_voice_steal

; round robin search for a free voice
NOTALLBUSY:
  MOV #0x10, W9                      ; 16 bits or round robin
RR3:
  SL ROUNDROBIN

; dsPIC uses 16 bits
  MOV ROUNDROBIN, W10
  CP0 W10
  BRA NZ, RR2

  MOV #0x0001, W10
  MOV W10, ROUNDROBIN
;
RR2:
; 16 bit...
  MOV ROUNDROBIN, W10
  MOV #0xFFFF, W0
	XOR _GATES, WREG                   ; XOR WREG with _GATES and store in WREG
	AND W10, W0, W0                    ; AND W0 with W10, store in W0
  BRA NZ, Pluck_It

  DEC W9, W9
	BRA NZ, RR3

; We shouldn't really ever get here, because we've checked if all voices are busy.
; but if we are, just do a simple round robin and steal  a voice.  

; assign with round robin when all voices are busy
; This algorithm will steal a voice that is already busy
RR_voice_steal:
  SL ROUNDROBIN

; 16 bit
  MOV ROUNDROBIN, W10
  CP0 W10
  BRA NZ, Pluck_It

; we are here if the 16 bit value for ROUNDROBIN is zero
  MOV #0x0001, W10
  MOV W10, ROUNDROBIN

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
Pluck_It:

; 1 bit.  
; attempt to find an unused voice
; there's only one place to go...
; send note data to string we will pluck
; W8 contains TRANSPOSE
PLUCK0:
  MOV DATAbyte1, W0
  MOV W0, _NOTE0                       ; save note value in SP RAM

  MOV W8, W0
  ADD DATAbyte1                        ; add TRANSPOSE

  MOV DATAbyte2, W0
  MOV W0, _VEL0                        ;

; look up the phase increment in the table using _NOTE0 as the address
; W1 has address into table
ChangePitch:

  MOV _NOTE0, W1
  SUB W1, #24, W1
;  SUB W1, #12, W1
  SL W1, W8                            ; mul by 2 because it's words
  MOV #tblpage(Tuning_Table), W2       ; Get table page address for the tuning table data
  MOV W2, TBLPAG                       ; Set the table page register here, all the data fits into the same page.
  MOV #tbloffset(Tuning_Table), W2     ; W0 gets the correct offset for TuningTable (which resides in Flash)
  ADD W2, W8, W1                       ; W1 gets the address into the tuning table 
  MOV #_PHINClo, W4
  TBLRDL [W1], [W4++]                  ; W0 gets the tuning ROM value for this note.
  TBLRDH [W1], [W4]
   
  ADD W8, #4, W8                       ; add 4 to the original note number

  MOV #tblpage(Tuning_Table), W2       ; Get table page address for the tuning table data
  MOV W2, TBLPAG                       ; Set the table page register here, all the data fits into the same page.
  MOV #tbloffset(Tuning_Table), W2     ; W0 gets the correct offset for TuningTable (which resides in Flash)
  ADD W2, W8, W1                       ; W1 gets the address into the tuning table 
  MOV #_PHINChi, W4
  TBLRDL [W1], [W4++]                  ; W0 gets the tuning ROM value for this note.
  TBLRDH [W1], [W4]
  
  MOV #0x0001, W1                      ; set gate flag bit 0, use string 0
  GOTO MANAGE_GATE_ON                  ; go sent the gate on signal
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; TEMP now contains a single bit which represents the gate to turn on.  
;
; First force the gate in question off, XOR the pluck bit in TEMP with GATES, then
; send the byte to gates.  If the GATES bit was on, it will go off.  If it was off
; it will go on.  In the second case, the second OUTPUT instruction doesn't matter

; 16 bits.  W1  (as TEMP) has gate bit to turn on
MANAGE_GATE_ON:
  MOV _GATES, W9                       ; get current gates state
  MOV W1, W10                          ; copy the gate bit to turn on into sE for inversion
  XOR W9, W10, W10                     ; XOR the pluck bit in TEMP with GATES
  IOR W9, W10, W9                      ; OR in the gate we want to turn on with the current gates status
  MOV W9, _GATES                       ; save new gates state

  GOTO idle

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

CONTROLLER_CHANGE:                           ; handle mod wheel, sustain and joystick here
  MOV #MOD_WHEEL, W0
  MOV DATAbyte1, W1
  CP W0, W1                                  ; mod wheel message?
  BRA NZ, tstCCsus                           ;
  MOV DATAbyte2, W0                          ; get CC message data
  MOV W0, _MOD_WHEEL                         ; store in _MOD_WHEEL
  GOTO idle                                  ;

tstCCsus:
  MOV #SUSTAIN, W0
  MOV DATAbyte1, W1
  CP W0, W1                                  ; is it a sustain message?
  BRA NZ, tstCCjoyx                          ;
  MOV DATAbyte2, W0
  MOV W0, SYNTH_SUSTAIN                      ; update synth
  MOV W0, SUS                                ; save sustain state, (maybe) need it for release logic
  GOTO idle                                  ;

tstCCjoyx:
  MOV #JOYSTICK_X, W0
  MOV DATAbyte1, W1
  CP W0, W1                                  ;
  BRA NZ, tstCCjoyy                          ;
  MOV DATAbyte2, W0
  MOV W0, _JOYSTICK_X                        ;
  GOTO idle                                  ; _JOYSTICK_X is defined in C

tstCCjoyy:  
  MOV #JOYSTICK_Y, W0
  MOV DATAbyte1, W1
  CP W0, W1                                  ;
  BRA NZ, tstCCallnotesoff                   ;
  MOV DATAbyte2, W0
  MOV W0, _JOYSTICK_Y                        ;
  GOTO idle                                  ; _JOYSTICK_Y is defined in C

tstCCallnotesoff:  
  MOV #ALL_NOTES_OFF, W0
  MOV DATAbyte1, W1
  CP W0, W1                                  ;
  BRA NZ, idle                               ; we don't recognize this CC
  CLR _GATES                                 ; clear _GATES
  CLR SYNTH_GATES                            ;  
  CLR W9                                     ; may contain a copy of _GATES

  GOTO idle

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

CHANNEL_PRESSURE:                            ; handle channel pressure here
  MOV DATAbyte2, W0
  MOV W0, SYNTH_CHANNEL_PRESSURE             ;
  GOTO idle                                  ;

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

PITCH_BEND:
  MOV DATAbyte1, W0                          ; low 7 bits of pitch wheel
  MOV DATAbyte2, W1                          ; high 7 bits of pitch wheel
  SL W1, #7, W1                              ; shift the high word left by 7
  ADD W0, W1, W0                             ; and add it to the low 7 bits in W0.  W0 now has an integer with the 14 bit pitch wheel value
  SL W0, #1, W0                              ; convert 14 bit value to fixed point
  MOV W0, _PITCH_WHEEL                       ; move to the RAM location for C
  GOTO idle                                  ;

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; IGNORE THESE MESSAGES
;; Here we just jump to the idle loop.  My keyboard can't send these, so I don't care.  Maybe you do?
PROGRAM_CHANGE:                              ; This synth doesn't react to this...
POLY_KEY_PRESSURE:                           ; This synth doesn't react to this...
  GOTO idle                                  ;

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;     
; Subroutines - some may not be needed, but are left here for troubleshooting

;HEXwordOUT:              ; print hex value of W3
;  MOV W3, W2
;  SWAP W2
;  CALL HEXbyteOUT
;  MOV W3, W2
;  CALL HEXbyteOUT
;  RETURN

;HEXbyteOUT:              ; print hex value of W2
;  MOV W2, W0
;  SWAP.B W0          ; swap nybbles in lower byte
;  AND #0x000F, W0
;  CP W0, #10
;  BRA GE, ALPH1
;  IOR #0x0030, W0
;  GOTO HBO1  
;ALPH1:
;  ADD #0x0037, W0
;HBO1:
;  MOV W0, W1
;  CALL TTYOUT
;  MOV W2, W0
;  AND #0x000F, W0
;  CP W0, #10
;  BRA GE, ALPH2
;  IOR #0x0030, W0
;  GOTO HBO2
;ALPH2:
;  ADD #0x0037, W0
;HBO2:  
;  MOV W0, W1
;  CALL TTYOUT
;  RETURN  

;TTYOUT:
;  BTST U1STA, #8     ; test TRMT bit "transmit shift register is empty"
;  BRA Z, TTYOUT      ; wait until this is high
;  MOV W1, U1TXREG    ; 
;  RETURN  

;GETMIDI:             ; get MIDI byte to W2
;  BTST U2STA, #0     ; test UR2DA in U2STA  (data available bit)
;  BRA Z,GETMIDI      ; wait until it goes high
;  MOV U2RXREG, W2    ; get the byte

;  RETURN

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; This tuning table starts at Bb.  This is to accomodate implementation of pitch 
; wheel code.  Pitch wheel is used as an interpolator slider between the note below
; and above the one you are sending in the MIDI note on message.

  .section *,address(0x10000),code

Tuning_Table:
  .pword 0x00205E
  .pword 0x00224B
  .pword 0x002455
  .pword 0x00267E
  .pword 0x0028C8
  .pword 0x002B34
  .pword 0x002DC6
  .pword 0x00307F
  .pword 0x003361
  .pword 0x00366F
  .pword 0x0039AC
  .pword 0x003D1A
  .pword 0x0040BC
  .pword 0x004495
  .pword 0x0048A9
  .pword 0x004CFC
  .pword 0x00518F
  .pword 0x005669
  .pword 0x005B8C
  .pword 0x0060FE
  .pword 0x0066C2
  .pword 0x006CDF
  .pword 0x007358
  .pword 0x007A34
  .pword 0x008178
  .pword 0x00892B
  .pword 0x009153
  .pword 0x0099F7
  .pword 0x00A31F
  .pword 0x00ACD2
  .pword 0x00B719
  .pword 0x00C1FC
  .pword 0x00CD85
  .pword 0x00D9BD
  .pword 0x00E6B0
  .pword 0x00F467
  .pword 0x0102F0
  .pword 0x011256
  .pword 0x0122A6
  .pword 0x0133EE
  .pword 0x01463E
  .pword 0x0159A4
  .pword 0x016E31
  .pword 0x0183F8
  .pword 0x019B09
  .pword 0x01B37B
  .pword 0x01CD60
  .pword 0x01E8CF
  .pword 0x0205E0
  .pword 0x0224AB
  .pword 0x02454B
  .pword 0x0267DC
  .pword 0x028C7B
  .pword 0x02B347
  .pword 0x02DC62
  .pword 0x0307EF
  .pword 0x033613
  .pword 0x0366F5
  .pword 0x039ABF
  .pword 0x03D19E
  .pword 0x040BC0
  .pword 0x044956
  .pword 0x048A97
  .pword 0x04CFB8
  .pword 0x0518F6
  .pword 0x05668F
  .pword 0x05B8C5
  .pword 0x060FDE
  .pword 0x066C26
  .pword 0x06CDEA
  .pword 0x07357E
  .pword 0x07A33C
  .pword 0x08177F
  .pword 0x0892AD
  .pword 0x09152D
  .pword 0x099F70
  .pword 0x0A31EC
  .pword 0x0ACD1E
  .pword 0x0B718A
  .pword 0x0C1FBD
  .pword 0x0CD84C
  .pword 0x0D9BD4
  .pword 0x0E6AFD
  .pword 0x0F4677
  .pword 0x102EFE
  .pword 0x112559
  .pword 0x122A5A
  .pword 0x133EE1
  .pword 0x1463D8
  .pword 0x16E314
  .pword 0x19B098
  .pword 0x1CD5FA
  .pword 0x205DFD
  .pword 0x2454B5
  .pword 0x28C7B0
.end

