.include "p33FJ128GP802.inc"

.data

LFSR:             .word 0x39F1
                  .word 0x9723
xor_bit:          .word 0x0000

port_sample_cnt:  .word 0x0000
port_temp:        .word 0x0000

.text

.global _saw24
.global _sine
.global _dist_sine
.global _sine_slow
.global _fastmul
.global _widemul
.global _ADSR
.global _noise
.global _seed
.global _lowpass1pole
.global _interp
.global _portamento

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; portamento
; inline extern void portamento( long *PhaseInc, int *PhaseIncPort, int Speed, int SkipSamples );
;                                     W0             W1                 W2         W3
; Portamento is supported via a single pole lowpass filter using 32 bit arithmetic.
; A sample counter is employed to reduce the sample rate for this filter to allow
; for a useful range of speeds.
; note that Speed is a CC value expressed as an integer.
; It must be converted to fixed point before it can be used.

;  port_speed = ((long)JOYSTICK_Y << 21) + 0x01000000L;         // set portamento speed
;  port_a0 = port_speed;
;  port_b1 = 0x7FFFFFFFL - port_speed;
;  ta0 = widemul( port_a0, PHINC0 );
;  tb1 = widemul( port_b1, PHINC0p );
;  PHINC0p = (unsigned long)(ta0 + tb1);

_portamento:
  CP0 port_sample_cnt
  BRA NZ, port_exit
  
  MOV W3, port_sample_cnt                 ; move SkipSamples to counter
  
  MOV #0x007F, W13
  CP W13, W2
  BRA NZ, port_cont                       ; if Speed == 127
  MOV [W0++], [W1++]                      ; transfer PhaseInc to PhaseIncPort
  MOV [W0], [W1]                          ; transfer PhaseInc to PhaseIncPort
  RETURN                                  ; and bail out

port_cont:
  PUSH W1                                 ; save address of PhaseIncPort onto stack

  SL W2, #5, W13                          ; W2 contains high word of Speed.  Low word is all zeroes.
  ADD #0x0100, W13                        ; keep Speed in W13

  MOV [W0++], W10                         ;
  MOV [W0], W11                           ; W11:W10 has PhaseInc

  MOV [W1++], W2                          ;
  MOV [W1--], W3                          ; W3:W2 has PhaseIncPort

; calc b1 from a0
  MOV #0xFFFF, W0                         ; low byte of Speed is zero
  MOV #0x7FFF, W4                         ; subtract high word of a0 from 0x7FFF
  SUB W4, W13, W1                         ; finish subtract.  W1:W0 has b1 val

; now mult b1 * PhaseIncPort  
;_widemul:
  MUL.SS    W1,W3,W8                      ; Calculate partial sum R4 = W9:W8 
  MUL.US    W0,W3,W6                      ; Calculate partial sum R3 = W7:W6 
  MUL.SU    W1,W2,W4                      ; Calculate partial sum R2 = W5:W4 
  MUL.UU    W0,W2,W0                      ; Calculate partial sum R1 = W1:W0 
 
  ADD       W1,W4,W1                      ; W3:W2:W1:W0 = W9:W8:W1:W0 
  ADDC      W5,W8,W2                      ;             + **:W5:W4:00 
  ADDC      W9,#0,W3 
  ADD       W1,W6,W1                      ; W3:W2:W1:W0 = W3:W2:W1:W0 
  ADDC      W2,W7,W2                      ;             + **:W7:W6:00 
  ADDC      #0,W3 
 
  BTSC      W5,#15                        ; If (R2 < 0), subtract 1 from the MSW 
  DEC       W3,W3 
  BTSC      W7,#15                        ; If (R3 < 0), subtract 1 from the MSW 
  DEC       W3,W3 
; now adjust for fixed point:  (result is in W3:W2:W1:W0)
  SL        W1, #1, W1                    ; move high bit to carry
  RLC       W2, W0                        ;
  RLC       W3, W1                        ; 32 bit fixed point result in W1:W0

  MOV W0, port_temp                       ; save result of b1 * PhaseIncPort
  MOV W1, port_temp+2

  MOV W10, W0                             ; get low word of PhaseInc
  MOV W11, W1                             ; get high word of PhaseInc

  CLR W2                                  ; get low word of Speed
  MOV W13, W3                             ; get high word of Speed

;_widemul:
  MUL.SS    W1,W3,W8                      ; Calculate partial sum R4 = W9:W8 
  MUL.US    W0,W3,W6                      ; Calculate partial sum R3 = W7:W6 
  MUL.SU    W1,W2,W4                      ; Calculate partial sum R2 = W5:W4 
  MUL.UU    W0,W2,W0                      ; Calculate partial sum R1 = W1:W0 
 
  ADD       W1,W4,W1                      ; W3:W2:W1:W0 = W9:W8:W1:W0 
  ADDC      W5,W8,W2                      ;             + **:W5:W4:00 
  ADDC      W9,#0,W3 
  ADD       W1,W6,W1                      ; W3:W2:W1:W0 = W3:W2:W1:W0 
  ADDC      W2,W7,W2                      ;             + **:W7:W6:00 
  ADDC      #0,W3 
 
  BTSC      W5,#15                        ; If (R2 < 0), subtract 1 from the MSW 
  DEC       W3,W3 
  BTSC      W7,#15                        ; If (R3 < 0), subtract 1 from the MSW 
  DEC       W3,W3 
; now adjust for fixed point:  (result is in W3:W2:W1:W0)
  SL        W1, #1, W1                    ; move high bit to carry
  RLC       W2, W0                        ;
  RLC       W3, W1                        ; 32 bit fixed point result in W1:W0

  POP W4                                  ; get address of PhaseIncPort
  MOV #port_temp, W5                      ; add the two terms to form the filter
  ADD W0, [W5++], [W4++]                  ; 
  ADDC W1, [W5], [W4]                     ; sum and store the result 
  RETURN
port_exit:
  DEC port_sample_cnt                     ; decrement the sample counter
  RETURN


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; Interpolator (linear) 16 bit
; int interp( int p1, int p2, int *slider_P, *slider_N );  OLD
;                 W0      W1      W2         W3
; int interp( int p1, int p2, int *slider );               NEW
;                 W0      W1      W2
_interp:
  MOV #0x8000, W3       ; 
  SUB W3, [W2], W3      ; W3 = 32768 - slider
  MUL.SS W0, [W2], W4   ; <W5,W4> = slider * p1
  SL W4, W4             ; fix fixed point magnitude
  RLC W5, W5            ; fix fixed point magnitude
  MUL.SS W1, W3, W6     ; <W7,W6> = W3 * p2
  SL W6, W6             ; fix fixed point magnitude
  RLC W7, W7            ; fix fixed point magnitude
  ADD W5, W7, W0        ;
  RETURN

/*
_interp:
  MOV #0x7FFF, W3       ; 
  SUB W3, [W2], W3      ; W3 = 32767 - slider
  MUL.SS W0, [W2], W4   ; <W5,W4> = slider * p1
  SL W4, W4             ; fix fixed point magnitude
  RLC W5, W5            ; fix fixed point magnitude
  MUL.SS W1, W3, W6     ; <W7,W6> = W3 * p2
  SL W6, W6             ; fix fixed point magnitude
  RLC W7, W7            ; fix fixed point magnitude
  ADD W5, W7, W0        ;
  RETURN
*/

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; single pole lowpass filter - 16 bit
; void lowpass1pole( int input, int *memory, int a0 );
;                        W0         W1           W2
_lowpass1pole:
  MUL.SS W2, W0, W4     ; input * a0
  SL W4, W4
  RLC W5, W6            ; a0 term, in W6
  MOV #32767, W7
  SUB W7, W2, W2        ; make b1 in W2
  MUL.SS W2, [W1], W4
  SL W4, W4
  RLC W5, W7            ; b1 term in W7
  ADD W6, W7, W0
  MOV W0, [W1]
  RETURN

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; 31 bit LFSR noise source.  16 bit output.
; bits 30 and 27 are used in xor feedback
; void noise( int * );
;
_noise:
  MOV #LFSR, W7         ; W7 points to low word of LFSR
  CLR xor_bit           ; the xor output bit
  BTSS LFSR+2, #14      ; the next 4 instructions are the XOR algorithm
  BTG xor_bit, #0       ;
  BTSS LFSR+2, #11      ;
  BTG xor_bit, #0       ; xor_bit has the xor output
  SL LFSR               ; left shit the low word, (bit 15 to C)
  RLC LFSR+2,           ; rotate left the high word
  MOV xor_bit, W6       ; get the xor output bit
  IOR W6, [W7], [W7]    ; or it into low word (LFSR)
  MOV [W7], [W0]        ; low word is the return value
  RETURN

_seed:                  ; a function to seed the random number generator
  MOV W0, LFSR
  MOV W1, LFSR+2
  RETURN

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; fast 16x16=16 signed fixed point multiply

_fastmul:
  MUL.SS W0, W1, W4
  SL W4, W4              ; high bit goes to C
  RLC W5, [W2]
  RETURN

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; wide fixed point multiplier.  32x32=32
; long widemul( long, long )
;               W1,W0 W3,W2

_widemul:
  MUL.SS    W1,W3,W8                      ; Calculate partial sum R4 = W9:W8 
  MUL.US    W0,W3,W6                      ; Calculate partial sum R3 = W7:W6 
  MUL.SU    W1,W2,W4                      ; Calculate partial sum R2 = W5:W4 
  MUL.UU    W0,W2,W0                      ; Calculate partial sum R1 = W1:W0 
 
  ADD       W1,W4,W1                      ; W3:W2:W1:W0 = W9:W8:W1:W0 
  ADDC      W5,W8,W2                      ;             + **:W5:W4:00 
  ADDC      W9,#0,W3 
  ADD       W1,W6,W1                      ; W3:W2:W1:W0 = W3:W2:W1:W0 
  ADDC      W2,W7,W2                      ;             + **:W7:W6:00 
  ADDC      #0,W3 
 
  BTSC      W5,#15                        ; If (R2 < 0), subtract 1 from the MSW 
  DEC       W3,W3 
  BTSC      W7,#15                        ; If (R3 < 0), subtract 1 from the MSW 
  DEC       W3,W3 
; now adjust for fixed point:  (result is in W3:W2:W1:W0)
  SL        W1, #1, W1                    ; move high bit to carry
  RLC       W2, W0                        ;
  RLC       W3, W1                        ; 32 bit fixed point result in W1:W0
  RETURN  

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; ADSR
;void ADSR( unsigned *state, int *value, int A, int D, int S, int R )
;                    W0          W1          W2     W3     W4     W5  
;
; A state machine which implements an ADSR

.equ IDLE, 0
.equ ATTACK, 1
.equ DECAY, 2
.equ SUSTAIN, 3
.equ RELEASE, 4

_ADSR:                 ; a branch tree to vector the states
  MOV [W0], W6
  CP0 W6
  BRA Z, ADSR_IDLE
  CP W6, #ATTACK
  BRA Z, ADSR_A
  CP W6, #DECAY
  BRA Z, ADSR_D
  CP W6, #SUSTAIN
  BRA Z, ADSR_S
  CP W6, #RELEASE
  BRA Z, ADSR_R
  BRA ADSR_IDLE

ADSR_IDLE:             ; this is the idle state, ADSR is not active.
  CP0 _GATES           ; test _GATES to see if a state change is necessary
  BRA Z, ADSR_ret      ; if _GATES == 0, then remain in idle state
  MOV #ATTACK, W6      ;
  MOV W6, [W0]         ; change state to ADSR_A (Attack)
  MOV W2, [W1]         ; set initial value to A
  RETURN

ADSR_A:
  CP0 _GATES           ;
  BRA NZ, ADSR_A1      ;
  MOV #RELEASE, W6     ; if _GATES is low...
  MOV W6, [W0]         ; change state to ADSR_R (Release)
  RETURN
ADSR_A1:
  ADD W2, [W1], [W1]   ; Add A rate value to output value
  BTST [W1], #15       ; did output value go negative?
  BRA Z, ADSR_A2       ;
  MOV #32767, W6       ; store max output value
  MOV W6, [W1]         ; in the value location
  MOV #DECAY, W6       ; go to Decay state next
  MOV W6, [W0]         ;
ADSR_A2:
  RETURN

ADSR_D:
  CP0 _GATES           ; 
  BRA NZ, ADSR_D1      ; if GATES is zero
  MOV #RELEASE, W6     ; 
  MOV W6, [W0]         ; change state to ADSR_R (Release)
  RETURN
ADSR_D1:
  MOV [W1], W6         ;
  SUB W6, W3, [W1]     ; *value = *value - D
  CP W4, [W1]          ; compare output with sustain value
  BRA LE, ADSR_D2      ;
  MOV W4, [W1]         ; *value = S    set output to sustain value
  MOV #SUSTAIN, W6     ;
  MOV W6, [W0]         ; change state to ADSR_S (Sustain)
ADSR_D2:
  RETURN

ADSR_S:
  MOV W4, [W1]         ; set output to sustain value
  CP0 _GATES           ; Is _GATES now low?
  BRA NZ, ADSR_S1      ;
  MOV #RELEASE, W6     ;
  MOV W6, [W0]         ; change state to ADSR_R (Release)
ADSR_S1:
  RETURN

ADSR_R:
  CP0 _GATES           ; is _GATES still low?
  BRA Z, ADSR_R1       ;
  MOV #ATTACK, W6      ; if _GATES went high, set Attack state
  MOV W6, [W0]         ; change state to ADSR_A (Attack)
  RETURN
ADSR_R1:
  MOV [W1], W6         ; get last output value to W6
  SUB W6, W5, [W1]     ; subtract R rate from output value
  BTST [W1], #15       ; did output go negative?
  BRA Z, ADSR_R2       ;
  CLR [W1]             ; if output is negative, set output to zero
  MOV #IDLE, W6        ; set ADSR idle state
  MOV W6, [W0]         ; change state to ADSR_IDLE (Idle)
ADSR_R2:
ADSR_ret:
  RETURN

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; saw24
;
; inline extern void saw24( unsigned long *PhaseAccum, unsigned long *PhaseIncrement );
;
; W0 has the address of the phase accumulator (32 bits unsigned)
; W1 has the address of the phase increment (32 bits unsigned)
;
; This function generates data for a 24 bit phase accumulator oscillator.
; The sawtooth output is taken directly from the 24 bit phase accumulator.
; Pitch is expressed with the value passed as a "phase increment" value, as a pointer 
; stored in W1 (the address of the phase accumulator).  Each time the function is called,
; the phase accumulator location is updated by adding the phase increment to it.
; 

_saw24:
  MOV [W0], W4           ; get phase accum low word
  MOV [W0+2], W5         ; get phase accum high word
  ADD W4, [W1++], [W0++] ; add phase increment to phase accum
  ADDC W5, [W1], [W0]    ; add with carry
  CLR.B [++W0]           ; force high byte to zeroes (24 bit unsigned phase accum.
  ; we want to return a 16 bit angle
  MOV.B [W0-1], W1       ; move the high byte to W1
  MOV.B [W0-2], W2       ; move the low byte to W2
  SL W1, #8, W1          ; shift the low byte into the high byte
  ADD W1, W2, W0         ; return the angle value via W0
  RETURN

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; Angle is in W0.  Uses a full wave table stored at sine_table
; W1 :: tri_tmp and tri_out
;
; This function returns the sine of the angle passed to it.  The
; angle is a number which can be from 0x0000 to 0xFFFF where
; 0x0000 is zero radians, 0x8000 is pi radians and 0xFFFF is 
; nearly 2Pi radians.  The function uses a 1/4 cycle table of 1024
; values (16 bits wide with 15 bit unsigned values).  The logic
; performs the unfolding required to give an output that is 16 bit
; significant and 4096 values per cycle.

_sine:     ; new version (4K element table, fast)
; W0 is a 12 bit address when shifted down 4 bits
  LSR W0, #4, W1
; W1 has address into table
  SL W1, W1                          ; word address
  MOV #tblpage(sine_table), W2       ; Get table page address for the tuning table data
  MOV W2, TBLPAG                     ; Set the table page register here, all the data fits into the same page.
  MOV #tbloffset(sine_table), W2     ; W0 gets the correct offset for TuningTable (which resides in Flash)
  ADD W2, W1, W0                     ; W0 gets the address into the tuning table 
  TBLRDL [W0], W0                    ; W0 gets the sine value from the table
  RETURN

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; Angle is in W0.  Uses a full wave table stored at distorted_sine_table
;
; This function returns a soft clipped sine.
;

_dist_sine:     ; new version (4K element table, fast)
; W0 is a 12 bit address when shifted down 4 bits
  LSR W0, #4, W1
; W1 has address into table
  SL W1, W1                                ; word address
  MOV #tblpage(distorted_sine_table), W2   ; Get table page address for the tuning table data
  MOV W2, TBLPAG                           ; Set the table page register here, all the data fits into the same page.
  MOV #tbloffset(distorted_sine_table), W2 ; W2 gets the correct offset for TuningTable (which resides in Flash)
  ADD W2, W1, W0                           ; W0 gets the address into the tuning table 
  TBLRDL [W0], W0                          ; W0 gets the sine value from the table
  RETURN

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; Angle is in W0.  Uses a 1/4 wave table stored at sine_table

_sine_slow:  ; old version (uses 1K element table, but is slower)
             ;              uses the first 1K elements of sine_table.
;  assign tri_tmp = PHacc[out_hi_bit:out_hi_bit] ? ~PHacc[out_hi_bit-1:out_hi_bit-18] : PHacc[out_hi_bit-1:out_hi_bit-18] ;
  BTST W0, #15           ; test high bit of angle (which is unsigned)
  BRA Z, sine1
  COM W0, W1             ; make a complemented copy of W0 in W1
  SL W1, W1              ; shift left 1.  this makes tri output full scale.
  GOTO sine2
sine1:
  SL W0, W1              ; don't complement, but still make tri output full scale.
sine2:
;  assign adr = tri_out[15] ? ~tri_out[14:5] : tri_out[14:5];        // invert address if tri is negative
  MOV W1, W3             ; W3 is tri_out, W1 becomes adr
  BTST W3, #15           ;
  BRA NZ, sine3          ;
  COM W1, W1             ; if the tri output value is negative, complement it
sine3:
  
  LSR W1, #5, W1         ; shift right 5 bits

  MOV #0x03FF, W2
  AND W1, W2, W1

; W1 has address into table
  SL W1, W1                          ; word address
  MOV #tblpage(sine_table), W2       ; Get table page address for the tuning table data
  MOV W2, TBLPAG                     ; Set the table page register here, all the data fits into the same page.
  MOV #tbloffset(sine_table), W2     ; W0 gets the correct offset for TuningTable (which resides in Flash)
  ADD W2, W1, W0                     ; W0 gets the address into the tuning table 
  TBLRDL [W0], W0                    ; W0 gets the sine value from the table

;  assign out = tri_out[17] ? -sinT : sinT ;                         // use only 17 bits for now, also use tri sign bit
  BTST W3, #15           ;
  BRA NZ, sine4
  NEG W0, W0
sine4:
  RETURN

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;.section *,address(0x10100),code
.section *,code

sine_table:
  .pword 0x0000
  .pword 0x0032
  .pword 0x0065
  .pword 0x0097
  .pword 0x00C9
  .pword 0x00FB
  .pword 0x012E
  .pword 0x0160
  .pword 0x0192
  .pword 0x01C4
  .pword 0x01F7
  .pword 0x0229
  .pword 0x025B
  .pword 0x028D
  .pword 0x02C0
  .pword 0x02F2
  .pword 0x0324
  .pword 0x0356
  .pword 0x0389
  .pword 0x03BB
  .pword 0x03ED
  .pword 0x041F
  .pword 0x0452
  .pword 0x0484
  .pword 0x04B6
  .pword 0x04E8
  .pword 0x051B
  .pword 0x054D
  .pword 0x057F
  .pword 0x05B1
  .pword 0x05E3
  .pword 0x0616
  .pword 0x0648
  .pword 0x067A
  .pword 0x06AC
  .pword 0x06DE
  .pword 0x0711
  .pword 0x0743
  .pword 0x0775
  .pword 0x07A7
  .pword 0x07D9
  .pword 0x080B
  .pword 0x083E
  .pword 0x0870
  .pword 0x08A2
  .pword 0x08D4
  .pword 0x0906
  .pword 0x0938
  .pword 0x096A
  .pword 0x099D
  .pword 0x09CF
  .pword 0x0A01
  .pword 0x0A33
  .pword 0x0A65
  .pword 0x0A97
  .pword 0x0AC9
  .pword 0x0AFB
  .pword 0x0B2D
  .pword 0x0B5F
  .pword 0x0B92
  .pword 0x0BC4
  .pword 0x0BF6
  .pword 0x0C28
  .pword 0x0C5A
  .pword 0x0C8C
  .pword 0x0CBE
  .pword 0x0CF0
  .pword 0x0D22
  .pword 0x0D54
  .pword 0x0D86
  .pword 0x0DB8
  .pword 0x0DEA
  .pword 0x0E1C
  .pword 0x0E4E
  .pword 0x0E80
  .pword 0x0EB1
  .pword 0x0EE3
  .pword 0x0F15
  .pword 0x0F47
  .pword 0x0F79
  .pword 0x0FAB
  .pword 0x0FDD
  .pword 0x100F
  .pword 0x1041
  .pword 0x1072
  .pword 0x10A4
  .pword 0x10D6
  .pword 0x1108
  .pword 0x113A
  .pword 0x116C
  .pword 0x119D
  .pword 0x11CF
  .pword 0x1201
  .pword 0x1233
  .pword 0x1264
  .pword 0x1296
  .pword 0x12C8
  .pword 0x12FA
  .pword 0x132B
  .pword 0x135D
  .pword 0x138F
  .pword 0x13C0
  .pword 0x13F2
  .pword 0x1424
  .pword 0x1455
  .pword 0x1487
  .pword 0x14B9
  .pword 0x14EA
  .pword 0x151C
  .pword 0x154D
  .pword 0x157F
  .pword 0x15B0
  .pword 0x15E2
  .pword 0x1613
  .pword 0x1645
  .pword 0x1676
  .pword 0x16A8
  .pword 0x16D9
  .pword 0x170B
  .pword 0x173C
  .pword 0x176E
  .pword 0x179F
  .pword 0x17D0
  .pword 0x1802
  .pword 0x1833
  .pword 0x1865
  .pword 0x1896
  .pword 0x18C7
  .pword 0x18F9
  .pword 0x192A
  .pword 0x195B
  .pword 0x198C
  .pword 0x19BE
  .pword 0x19EF
  .pword 0x1A20
  .pword 0x1A51
  .pword 0x1A82
  .pword 0x1AB4
  .pword 0x1AE5
  .pword 0x1B16
  .pword 0x1B47
  .pword 0x1B78
  .pword 0x1BA9
  .pword 0x1BDA
  .pword 0x1C0B
  .pword 0x1C3C
  .pword 0x1C6D
  .pword 0x1C9E
  .pword 0x1CCF
  .pword 0x1D00
  .pword 0x1D31
  .pword 0x1D62
  .pword 0x1D93
  .pword 0x1DC4
  .pword 0x1DF5
  .pword 0x1E26
  .pword 0x1E57
  .pword 0x1E87
  .pword 0x1EB8
  .pword 0x1EE9
  .pword 0x1F1A
  .pword 0x1F4A
  .pword 0x1F7B
  .pword 0x1FAC
  .pword 0x1FDD
  .pword 0x200D
  .pword 0x203E
  .pword 0x206F
  .pword 0x209F
  .pword 0x20D0
  .pword 0x2100
  .pword 0x2131
  .pword 0x2161
  .pword 0x2192
  .pword 0x21C2
  .pword 0x21F3
  .pword 0x2223
  .pword 0x2254
  .pword 0x2284
  .pword 0x22B5
  .pword 0x22E5
  .pword 0x2315
  .pword 0x2346
  .pword 0x2376
  .pword 0x23A6
  .pword 0x23D7
  .pword 0x2407
  .pword 0x2437
  .pword 0x2467
  .pword 0x2497
  .pword 0x24C8
  .pword 0x24F8
  .pword 0x2528
  .pword 0x2558
  .pword 0x2588
  .pword 0x25B8
  .pword 0x25E8
  .pword 0x2618
  .pword 0x2648
  .pword 0x2678
  .pword 0x26A8
  .pword 0x26D8
  .pword 0x2708
  .pword 0x2737
  .pword 0x2767
  .pword 0x2797
  .pword 0x27C7
  .pword 0x27F7
  .pword 0x2826
  .pword 0x2856
  .pword 0x2886
  .pword 0x28B5
  .pword 0x28E5
  .pword 0x2915
  .pword 0x2944
  .pword 0x2974
  .pword 0x29A3
  .pword 0x29D3
  .pword 0x2A02
  .pword 0x2A32
  .pword 0x2A61
  .pword 0x2A91
  .pword 0x2AC0
  .pword 0x2AF0
  .pword 0x2B1F
  .pword 0x2B4E
  .pword 0x2B7D
  .pword 0x2BAD
  .pword 0x2BDC
  .pword 0x2C0B
  .pword 0x2C3A
  .pword 0x2C6A
  .pword 0x2C99
  .pword 0x2CC8
  .pword 0x2CF7
  .pword 0x2D26
  .pword 0x2D55
  .pword 0x2D84
  .pword 0x2DB3
  .pword 0x2DE2
  .pword 0x2E11
  .pword 0x2E40
  .pword 0x2E6E
  .pword 0x2E9D
  .pword 0x2ECC
  .pword 0x2EFB
  .pword 0x2F2A
  .pword 0x2F58
  .pword 0x2F87
  .pword 0x2FB6
  .pword 0x2FE4
  .pword 0x3013
  .pword 0x3041
  .pword 0x3070
  .pword 0x309E
  .pword 0x30CD
  .pword 0x30FB
  .pword 0x312A
  .pword 0x3158
  .pword 0x3187
  .pword 0x31B5
  .pword 0x31E3
  .pword 0x3211
  .pword 0x3240
  .pword 0x326E
  .pword 0x329C
  .pword 0x32CA
  .pword 0x32F8
  .pword 0x3326
  .pword 0x3355
  .pword 0x3383
  .pword 0x33B1
  .pword 0x33DF
  .pword 0x340C
  .pword 0x343A
  .pword 0x3468
  .pword 0x3496
  .pword 0x34C4
  .pword 0x34F2
  .pword 0x351F
  .pword 0x354D
  .pword 0x357B
  .pword 0x35A8
  .pword 0x35D6
  .pword 0x3604
  .pword 0x3631
  .pword 0x365F
  .pword 0x368C
  .pword 0x36BA
  .pword 0x36E7
  .pword 0x3715
  .pword 0x3742
  .pword 0x376F
  .pword 0x379C
  .pword 0x37CA
  .pword 0x37F7
  .pword 0x3824
  .pword 0x3851
  .pword 0x387E
  .pword 0x38AB
  .pword 0x38D9
  .pword 0x3906
  .pword 0x3933
  .pword 0x3960
  .pword 0x398C
  .pword 0x39B9
  .pword 0x39E6
  .pword 0x3A13
  .pword 0x3A40
  .pword 0x3A6C
  .pword 0x3A99
  .pword 0x3AC6
  .pword 0x3AF2
  .pword 0x3B1F
  .pword 0x3B4C
  .pword 0x3B78
  .pword 0x3BA5
  .pword 0x3BD1
  .pword 0x3BFE
  .pword 0x3C2A
  .pword 0x3C56
  .pword 0x3C83
  .pword 0x3CAF
  .pword 0x3CDB
  .pword 0x3D07
  .pword 0x3D33
  .pword 0x3D60
  .pword 0x3D8C
  .pword 0x3DB8
  .pword 0x3DE4
  .pword 0x3E10
  .pword 0x3E3C
  .pword 0x3E68
  .pword 0x3E93
  .pword 0x3EBF
  .pword 0x3EEB
  .pword 0x3F17
  .pword 0x3F43
  .pword 0x3F6E
  .pword 0x3F9A
  .pword 0x3FC5
  .pword 0x3FF1
  .pword 0x401D
  .pword 0x4048
  .pword 0x4073
  .pword 0x409F
  .pword 0x40CA
  .pword 0x40F6
  .pword 0x4121
  .pword 0x414C
  .pword 0x4177
  .pword 0x41A2
  .pword 0x41CE
  .pword 0x41F9
  .pword 0x4224
  .pword 0x424F
  .pword 0x427A
  .pword 0x42A5
  .pword 0x42D0
  .pword 0x42FA
  .pword 0x4325
  .pword 0x4350
  .pword 0x437B
  .pword 0x43A5
  .pword 0x43D0
  .pword 0x43FB
  .pword 0x4425
  .pword 0x4450
  .pword 0x447A
  .pword 0x44A5
  .pword 0x44CF
  .pword 0x44F9
  .pword 0x4524
  .pword 0x454E
  .pword 0x4578
  .pword 0x45A3
  .pword 0x45CD
  .pword 0x45F7
  .pword 0x4621
  .pword 0x464B
  .pword 0x4675
  .pword 0x469F
  .pword 0x46C9
  .pword 0x46F3
  .pword 0x471C
  .pword 0x4746
  .pword 0x4770
  .pword 0x479A
  .pword 0x47C3
  .pword 0x47ED
  .pword 0x4816
  .pword 0x4840
  .pword 0x4869
  .pword 0x4893
  .pword 0x48BC
  .pword 0x48E5
  .pword 0x490F
  .pword 0x4938
  .pword 0x4961
  .pword 0x498A
  .pword 0x49B4
  .pword 0x49DD
  .pword 0x4A06
  .pword 0x4A2F
  .pword 0x4A58
  .pword 0x4A80
  .pword 0x4AA9
  .pword 0x4AD2
  .pword 0x4AFB
  .pword 0x4B24
  .pword 0x4B4C
  .pword 0x4B75
  .pword 0x4B9D
  .pword 0x4BC6
  .pword 0x4BEE
  .pword 0x4C17
  .pword 0x4C3F
  .pword 0x4C68
  .pword 0x4C90
  .pword 0x4CB8
  .pword 0x4CE0
  .pword 0x4D09
  .pword 0x4D31
  .pword 0x4D59
  .pword 0x4D81
  .pword 0x4DA9
  .pword 0x4DD1
  .pword 0x4DF9
  .pword 0x4E20
  .pword 0x4E48
  .pword 0x4E70
  .pword 0x4E98
  .pword 0x4EBF
  .pword 0x4EE7
  .pword 0x4F0E
  .pword 0x4F36
  .pword 0x4F5D
  .pword 0x4F85
  .pword 0x4FAC
  .pword 0x4FD4
  .pword 0x4FFB
  .pword 0x5022
  .pword 0x5049
  .pword 0x5070
  .pword 0x5097
  .pword 0x50BE
  .pword 0x50E5
  .pword 0x510C
  .pword 0x5133
  .pword 0x515A
  .pword 0x5181
  .pword 0x51A8
  .pword 0x51CE
  .pword 0x51F5
  .pword 0x521B
  .pword 0x5242
  .pword 0x5268
  .pword 0x528F
  .pword 0x52B5
  .pword 0x52DC
  .pword 0x5302
  .pword 0x5328
  .pword 0x534E
  .pword 0x5374
  .pword 0x539B
  .pword 0x53C1
  .pword 0x53E7
  .pword 0x540C
  .pword 0x5432
  .pword 0x5458
  .pword 0x547E
  .pword 0x54A4
  .pword 0x54C9
  .pword 0x54EF
  .pword 0x5515
  .pword 0x553A
  .pword 0x5560
  .pword 0x5585
  .pword 0x55AA
  .pword 0x55D0
  .pword 0x55F5
  .pword 0x561A
  .pword 0x563F
  .pword 0x5664
  .pword 0x568A
  .pword 0x56AF
  .pword 0x56D3
  .pword 0x56F8
  .pword 0x571D
  .pword 0x5742
  .pword 0x5767
  .pword 0x578B
  .pword 0x57B0
  .pword 0x57D5
  .pword 0x57F9
  .pword 0x581E
  .pword 0x5842
  .pword 0x5867
  .pword 0x588B
  .pword 0x58AF
  .pword 0x58D3
  .pword 0x58F8
  .pword 0x591C
  .pword 0x5940
  .pword 0x5964
  .pword 0x5988
  .pword 0x59AC
  .pword 0x59CF
  .pword 0x59F3
  .pword 0x5A17
  .pword 0x5A3B
  .pword 0x5A5E
  .pword 0x5A82
  .pword 0x5AA5
  .pword 0x5AC9
  .pword 0x5AEC
  .pword 0x5B0F
  .pword 0x5B33
  .pword 0x5B56
  .pword 0x5B79
  .pword 0x5B9C
  .pword 0x5BBF
  .pword 0x5BE2
  .pword 0x5C05
  .pword 0x5C28
  .pword 0x5C4B
  .pword 0x5C6E
  .pword 0x5C91
  .pword 0x5CB3
  .pword 0x5CD6
  .pword 0x5CF9
  .pword 0x5D1B
  .pword 0x5D3E
  .pword 0x5D60
  .pword 0x5D82
  .pword 0x5DA5
  .pword 0x5DC7
  .pword 0x5DE9
  .pword 0x5E0B
  .pword 0x5E2D
  .pword 0x5E4F
  .pword 0x5E71
  .pword 0x5E93
  .pword 0x5EB5
  .pword 0x5ED7
  .pword 0x5EF8
  .pword 0x5F1A
  .pword 0x5F3C
  .pword 0x5F5D
  .pword 0x5F7F
  .pword 0x5FA0
  .pword 0x5FC2
  .pword 0x5FE3
  .pword 0x6004
  .pword 0x6025
  .pword 0x6047
  .pword 0x6068
  .pword 0x6089
  .pword 0x60AA
  .pword 0x60CB
  .pword 0x60EB
  .pword 0x610C
  .pword 0x612D
  .pword 0x614E
  .pword 0x616E
  .pword 0x618F
  .pword 0x61AF
  .pword 0x61D0
  .pword 0x61F0
  .pword 0x6211
  .pword 0x6231
  .pword 0x6251
  .pword 0x6271
  .pword 0x6291
  .pword 0x62B1
  .pword 0x62D1
  .pword 0x62F1
  .pword 0x6311
  .pword 0x6331
  .pword 0x6351
  .pword 0x6370
  .pword 0x6390
  .pword 0x63AF
  .pword 0x63CF
  .pword 0x63EE
  .pword 0x640E
  .pword 0x642D
  .pword 0x644C
  .pword 0x646C
  .pword 0x648B
  .pword 0x64AA
  .pword 0x64C9
  .pword 0x64E8
  .pword 0x6507
  .pword 0x6525
  .pword 0x6544
  .pword 0x6563
  .pword 0x6582
  .pword 0x65A0
  .pword 0x65BF
  .pword 0x65DD
  .pword 0x65FC
  .pword 0x661A
  .pword 0x6638
  .pword 0x6656
  .pword 0x6675
  .pword 0x6693
  .pword 0x66B1
  .pword 0x66CF
  .pword 0x66ED
  .pword 0x670A
  .pword 0x6728
  .pword 0x6746
  .pword 0x6764
  .pword 0x6781
  .pword 0x679F
  .pword 0x67BC
  .pword 0x67DA
  .pword 0x67F7
  .pword 0x6814
  .pword 0x6832
  .pword 0x684F
  .pword 0x686C
  .pword 0x6889
  .pword 0x68A6
  .pword 0x68C3
  .pword 0x68E0
  .pword 0x68FC
  .pword 0x6919
  .pword 0x6936
  .pword 0x6952
  .pword 0x696F
  .pword 0x698B
  .pword 0x69A8
  .pword 0x69C4
  .pword 0x69E0
  .pword 0x69FD
  .pword 0x6A19
  .pword 0x6A35
  .pword 0x6A51
  .pword 0x6A6D
  .pword 0x6A89
  .pword 0x6AA4
  .pword 0x6AC0
  .pword 0x6ADC
  .pword 0x6AF8
  .pword 0x6B13
  .pword 0x6B2F
  .pword 0x6B4A
  .pword 0x6B65
  .pword 0x6B81
  .pword 0x6B9C
  .pword 0x6BB7
  .pword 0x6BD2
  .pword 0x6BED
  .pword 0x6C08
  .pword 0x6C23
  .pword 0x6C3E
  .pword 0x6C59
  .pword 0x6C74
  .pword 0x6C8E
  .pword 0x6CA9
  .pword 0x6CC3
  .pword 0x6CDE
  .pword 0x6CF8
  .pword 0x6D13
  .pword 0x6D2D
  .pword 0x6D47
  .pword 0x6D61
  .pword 0x6D7B
  .pword 0x6D95
  .pword 0x6DAF
  .pword 0x6DC9
  .pword 0x6DE3
  .pword 0x6DFD
  .pword 0x6E16
  .pword 0x6E30
  .pword 0x6E4A
  .pword 0x6E63
  .pword 0x6E7C
  .pword 0x6E96
  .pword 0x6EAF
  .pword 0x6EC8
  .pword 0x6EE1
  .pword 0x6EFB
  .pword 0x6F14
  .pword 0x6F2C
  .pword 0x6F45
  .pword 0x6F5E
  .pword 0x6F77
  .pword 0x6F90
  .pword 0x6FA8
  .pword 0x6FC1
  .pword 0x6FD9
  .pword 0x6FF2
  .pword 0x700A
  .pword 0x7022
  .pword 0x703A
  .pword 0x7053
  .pword 0x706B
  .pword 0x7083
  .pword 0x709B
  .pword 0x70B2
  .pword 0x70CA
  .pword 0x70E2
  .pword 0x70FA
  .pword 0x7111
  .pword 0x7129
  .pword 0x7140
  .pword 0x7158
  .pword 0x716F
  .pword 0x7186
  .pword 0x719D
  .pword 0x71B4
  .pword 0x71CB
  .pword 0x71E2
  .pword 0x71F9
  .pword 0x7210
  .pword 0x7227
  .pword 0x723E
  .pword 0x7254
  .pword 0x726B
  .pword 0x7281
  .pword 0x7298
  .pword 0x72AE
  .pword 0x72C4
  .pword 0x72DB
  .pword 0x72F1
  .pword 0x7307
  .pword 0x731D
  .pword 0x7333
  .pword 0x7349
  .pword 0x735E
  .pword 0x7374
  .pword 0x738A
  .pword 0x739F
  .pword 0x73B5
  .pword 0x73CA
  .pword 0x73E0
  .pword 0x73F5
  .pword 0x740A
  .pword 0x7420
  .pword 0x7435
  .pword 0x744A
  .pword 0x745F
  .pword 0x7474
  .pword 0x7488
  .pword 0x749D
  .pword 0x74B2
  .pword 0x74C6
  .pword 0x74DB
  .pword 0x74F0
  .pword 0x7504
  .pword 0x7518
  .pword 0x752D
  .pword 0x7541
  .pword 0x7555
  .pword 0x7569
  .pword 0x757D
  .pword 0x7591
  .pword 0x75A5
  .pword 0x75B8
  .pword 0x75CC
  .pword 0x75E0
  .pword 0x75F3
  .pword 0x7607
  .pword 0x761A
  .pword 0x762D
  .pword 0x7641
  .pword 0x7654
  .pword 0x7667
  .pword 0x767A
  .pword 0x768D
  .pword 0x76A0
  .pword 0x76B3
  .pword 0x76C6
  .pword 0x76D8
  .pword 0x76EB
  .pword 0x76FE
  .pword 0x7710
  .pword 0x7722
  .pword 0x7735
  .pword 0x7747
  .pword 0x7759
  .pword 0x776B
  .pword 0x777D
  .pword 0x778F
  .pword 0x77A1
  .pword 0x77B3
  .pword 0x77C5
  .pword 0x77D7
  .pword 0x77E8
  .pword 0x77FA
  .pword 0x780B
  .pword 0x781D
  .pword 0x782E
  .pword 0x783F
  .pword 0x7850
  .pword 0x7862
  .pword 0x7873
  .pword 0x7884
  .pword 0x7894
  .pword 0x78A5
  .pword 0x78B6
  .pword 0x78C7
  .pword 0x78D7
  .pword 0x78E8
  .pword 0x78F8
  .pword 0x7909
  .pword 0x7919
  .pword 0x7929
  .pword 0x7939
  .pword 0x794A
  .pword 0x795A
  .pword 0x796A
  .pword 0x7979
  .pword 0x7989
  .pword 0x7999
  .pword 0x79A9
  .pword 0x79B8
  .pword 0x79C8
  .pword 0x79D7
  .pword 0x79E6
  .pword 0x79F6
  .pword 0x7A05
  .pword 0x7A14
  .pword 0x7A23
  .pword 0x7A32
  .pword 0x7A41
  .pword 0x7A50
  .pword 0x7A5F
  .pword 0x7A6D
  .pword 0x7A7C
  .pword 0x7A8B
  .pword 0x7A99
  .pword 0x7AA8
  .pword 0x7AB6
  .pword 0x7AC4
  .pword 0x7AD2
  .pword 0x7AE0
  .pword 0x7AEE
  .pword 0x7AFC
  .pword 0x7B0A
  .pword 0x7B18
  .pword 0x7B26
  .pword 0x7B33
  .pword 0x7B41
  .pword 0x7B4F
  .pword 0x7B5C
  .pword 0x7B69
  .pword 0x7B77
  .pword 0x7B84
  .pword 0x7B91
  .pword 0x7B9E
  .pword 0x7BAB
  .pword 0x7BB8
  .pword 0x7BC5
  .pword 0x7BD2
  .pword 0x7BDE
  .pword 0x7BEB
  .pword 0x7BF8
  .pword 0x7C04
  .pword 0x7C10
  .pword 0x7C1D
  .pword 0x7C29
  .pword 0x7C35
  .pword 0x7C41
  .pword 0x7C4D
  .pword 0x7C59
  .pword 0x7C65
  .pword 0x7C71
  .pword 0x7C7D
  .pword 0x7C88
  .pword 0x7C94
  .pword 0x7C9F
  .pword 0x7CAB
  .pword 0x7CB6
  .pword 0x7CC1
  .pword 0x7CCD
  .pword 0x7CD8
  .pword 0x7CE3
  .pword 0x7CEE
  .pword 0x7CF9
  .pword 0x7D04
  .pword 0x7D0E
  .pword 0x7D19
  .pword 0x7D24
  .pword 0x7D2E
  .pword 0x7D39
  .pword 0x7D43
  .pword 0x7D4D
  .pword 0x7D57
  .pword 0x7D62
  .pword 0x7D6C
  .pword 0x7D76
  .pword 0x7D80
  .pword 0x7D89
  .pword 0x7D93
  .pword 0x7D9D
  .pword 0x7DA6
  .pword 0x7DB0
  .pword 0x7DB9
  .pword 0x7DC3
  .pword 0x7DCC
  .pword 0x7DD5
  .pword 0x7DDF
  .pword 0x7DE8
  .pword 0x7DF1
  .pword 0x7DFA
  .pword 0x7E02
  .pword 0x7E0B
  .pword 0x7E14
  .pword 0x7E1D
  .pword 0x7E25
  .pword 0x7E2E
  .pword 0x7E36
  .pword 0x7E3E
  .pword 0x7E47
  .pword 0x7E4F
  .pword 0x7E57
  .pword 0x7E5F
  .pword 0x7E67
  .pword 0x7E6F
  .pword 0x7E77
  .pword 0x7E7E
  .pword 0x7E86
  .pword 0x7E8D
  .pword 0x7E95
  .pword 0x7E9C
  .pword 0x7EA4
  .pword 0x7EAB
  .pword 0x7EB2
  .pword 0x7EB9
  .pword 0x7EC0
  .pword 0x7EC7
  .pword 0x7ECE
  .pword 0x7ED5
  .pword 0x7EDC
  .pword 0x7EE2
  .pword 0x7EE9
  .pword 0x7EEF
  .pword 0x7EF6
  .pword 0x7EFC
  .pword 0x7F02
  .pword 0x7F09
  .pword 0x7F0F
  .pword 0x7F15
  .pword 0x7F1B
  .pword 0x7F21
  .pword 0x7F26
  .pword 0x7F2C
  .pword 0x7F32
  .pword 0x7F37
  .pword 0x7F3D
  .pword 0x7F42
  .pword 0x7F48
  .pword 0x7F4D
  .pword 0x7F52
  .pword 0x7F57
  .pword 0x7F5C
  .pword 0x7F61
  .pword 0x7F66
  .pword 0x7F6B
  .pword 0x7F70
  .pword 0x7F74
  .pword 0x7F79
  .pword 0x7F7D
  .pword 0x7F82
  .pword 0x7F86
  .pword 0x7F8A
  .pword 0x7F8F
  .pword 0x7F93
  .pword 0x7F97
  .pword 0x7F9B
  .pword 0x7F9F
  .pword 0x7FA2
  .pword 0x7FA6
  .pword 0x7FAA
  .pword 0x7FAD
  .pword 0x7FB1
  .pword 0x7FB4
  .pword 0x7FB8
  .pword 0x7FBB
  .pword 0x7FBE
  .pword 0x7FC1
  .pword 0x7FC4
  .pword 0x7FC7
  .pword 0x7FCA
  .pword 0x7FCD
  .pword 0x7FD0
  .pword 0x7FD2
  .pword 0x7FD5
  .pword 0x7FD8
  .pword 0x7FDA
  .pword 0x7FDC
  .pword 0x7FDF
  .pword 0x7FE1
  .pword 0x7FE3
  .pword 0x7FE5
  .pword 0x7FE7
  .pword 0x7FE9
  .pword 0x7FEB
  .pword 0x7FEC
  .pword 0x7FEE
  .pword 0x7FF0
  .pword 0x7FF1
  .pword 0x7FF3
  .pword 0x7FF4
  .pword 0x7FF5
  .pword 0x7FF6
  .pword 0x7FF7
  .pword 0x7FF8
  .pword 0x7FF9
  .pword 0x7FFA
  .pword 0x7FFB
  .pword 0x7FFC
  .pword 0x7FFD
  .pword 0x7FFD
  .pword 0x7FFE
  .pword 0x7FFE
  .pword 0x7FFE
  .pword 0x7FFF
  .pword 0x7FFF
  .pword 0x7FFF
  .pword 0x7FFF
  .pword 0x7FFF
  .pword 0x7FFF
  .pword 0x7FFF
  .pword 0x7FFE
  .pword 0x7FFE
  .pword 0x7FFE
  .pword 0x7FFD
  .pword 0x7FFD
  .pword 0x7FFC
  .pword 0x7FFB
  .pword 0x7FFA
  .pword 0x7FF9
  .pword 0x7FF8
  .pword 0x7FF7
  .pword 0x7FF6
  .pword 0x7FF5
  .pword 0x7FF4
  .pword 0x7FF3
  .pword 0x7FF1
  .pword 0x7FF0
  .pword 0x7FEE
  .pword 0x7FEC
  .pword 0x7FEB
  .pword 0x7FE9
  .pword 0x7FE7
  .pword 0x7FE5
  .pword 0x7FE3
  .pword 0x7FE1
  .pword 0x7FDF
  .pword 0x7FDC
  .pword 0x7FDA
  .pword 0x7FD8
  .pword 0x7FD5
  .pword 0x7FD2
  .pword 0x7FD0
  .pword 0x7FCD
  .pword 0x7FCA
  .pword 0x7FC7
  .pword 0x7FC4
  .pword 0x7FC1
  .pword 0x7FBE
  .pword 0x7FBB
  .pword 0x7FB8
  .pword 0x7FB4
  .pword 0x7FB1
  .pword 0x7FAD
  .pword 0x7FAA
  .pword 0x7FA6
  .pword 0x7FA2
  .pword 0x7F9F
  .pword 0x7F9B
  .pword 0x7F97
  .pword 0x7F93
  .pword 0x7F8F
  .pword 0x7F8A
  .pword 0x7F86
  .pword 0x7F82
  .pword 0x7F7D
  .pword 0x7F79
  .pword 0x7F74
  .pword 0x7F70
  .pword 0x7F6B
  .pword 0x7F66
  .pword 0x7F61
  .pword 0x7F5C
  .pword 0x7F57
  .pword 0x7F52
  .pword 0x7F4D
  .pword 0x7F48
  .pword 0x7F42
  .pword 0x7F3D
  .pword 0x7F37
  .pword 0x7F32
  .pword 0x7F2C
  .pword 0x7F26
  .pword 0x7F21
  .pword 0x7F1B
  .pword 0x7F15
  .pword 0x7F0F
  .pword 0x7F09
  .pword 0x7F02
  .pword 0x7EFC
  .pword 0x7EF6
  .pword 0x7EEF
  .pword 0x7EE9
  .pword 0x7EE2
  .pword 0x7EDC
  .pword 0x7ED5
  .pword 0x7ECE
  .pword 0x7EC7
  .pword 0x7EC0
  .pword 0x7EB9
  .pword 0x7EB2
  .pword 0x7EAB
  .pword 0x7EA4
  .pword 0x7E9C
  .pword 0x7E95
  .pword 0x7E8D
  .pword 0x7E86
  .pword 0x7E7E
  .pword 0x7E77
  .pword 0x7E6F
  .pword 0x7E67
  .pword 0x7E5F
  .pword 0x7E57
  .pword 0x7E4F
  .pword 0x7E47
  .pword 0x7E3E
  .pword 0x7E36
  .pword 0x7E2E
  .pword 0x7E25
  .pword 0x7E1D
  .pword 0x7E14
  .pword 0x7E0B
  .pword 0x7E02
  .pword 0x7DFA
  .pword 0x7DF1
  .pword 0x7DE8
  .pword 0x7DDF
  .pword 0x7DD5
  .pword 0x7DCC
  .pword 0x7DC3
  .pword 0x7DB9
  .pword 0x7DB0
  .pword 0x7DA6
  .pword 0x7D9D
  .pword 0x7D93
  .pword 0x7D89
  .pword 0x7D80
  .pword 0x7D76
  .pword 0x7D6C
  .pword 0x7D62
  .pword 0x7D57
  .pword 0x7D4D
  .pword 0x7D43
  .pword 0x7D39
  .pword 0x7D2E
  .pword 0x7D24
  .pword 0x7D19
  .pword 0x7D0E
  .pword 0x7D04
  .pword 0x7CF9
  .pword 0x7CEE
  .pword 0x7CE3
  .pword 0x7CD8
  .pword 0x7CCD
  .pword 0x7CC1
  .pword 0x7CB6
  .pword 0x7CAB
  .pword 0x7C9F
  .pword 0x7C94
  .pword 0x7C88
  .pword 0x7C7D
  .pword 0x7C71
  .pword 0x7C65
  .pword 0x7C59
  .pword 0x7C4D
  .pword 0x7C41
  .pword 0x7C35
  .pword 0x7C29
  .pword 0x7C1D
  .pword 0x7C10
  .pword 0x7C04
  .pword 0x7BF8
  .pword 0x7BEB
  .pword 0x7BDE
  .pword 0x7BD2
  .pword 0x7BC5
  .pword 0x7BB8
  .pword 0x7BAB
  .pword 0x7B9E
  .pword 0x7B91
  .pword 0x7B84
  .pword 0x7B77
  .pword 0x7B69
  .pword 0x7B5C
  .pword 0x7B4F
  .pword 0x7B41
  .pword 0x7B33
  .pword 0x7B26
  .pword 0x7B18
  .pword 0x7B0A
  .pword 0x7AFC
  .pword 0x7AEE
  .pword 0x7AE0
  .pword 0x7AD2
  .pword 0x7AC4
  .pword 0x7AB6
  .pword 0x7AA8
  .pword 0x7A99
  .pword 0x7A8B
  .pword 0x7A7C
  .pword 0x7A6D
  .pword 0x7A5F
  .pword 0x7A50
  .pword 0x7A41
  .pword 0x7A32
  .pword 0x7A23
  .pword 0x7A14
  .pword 0x7A05
  .pword 0x79F6
  .pword 0x79E6
  .pword 0x79D7
  .pword 0x79C8
  .pword 0x79B8
  .pword 0x79A9
  .pword 0x7999
  .pword 0x7989
  .pword 0x7979
  .pword 0x796A
  .pword 0x795A
  .pword 0x794A
  .pword 0x7939
  .pword 0x7929
  .pword 0x7919
  .pword 0x7909
  .pword 0x78F8
  .pword 0x78E8
  .pword 0x78D7
  .pword 0x78C7
  .pword 0x78B6
  .pword 0x78A5
  .pword 0x7894
  .pword 0x7884
  .pword 0x7873
  .pword 0x7862
  .pword 0x7850
  .pword 0x783F
  .pword 0x782E
  .pword 0x781D
  .pword 0x780B
  .pword 0x77FA
  .pword 0x77E8
  .pword 0x77D7
  .pword 0x77C5
  .pword 0x77B3
  .pword 0x77A1
  .pword 0x778F
  .pword 0x777D
  .pword 0x776B
  .pword 0x7759
  .pword 0x7747
  .pword 0x7735
  .pword 0x7722
  .pword 0x7710
  .pword 0x76FE
  .pword 0x76EB
  .pword 0x76D8
  .pword 0x76C6
  .pword 0x76B3
  .pword 0x76A0
  .pword 0x768D
  .pword 0x767A
  .pword 0x7667
  .pword 0x7654
  .pword 0x7641
  .pword 0x762D
  .pword 0x761A
  .pword 0x7607
  .pword 0x75F3
  .pword 0x75E0
  .pword 0x75CC
  .pword 0x75B8
  .pword 0x75A5
  .pword 0x7591
  .pword 0x757D
  .pword 0x7569
  .pword 0x7555
  .pword 0x7541
  .pword 0x752D
  .pword 0x7518
  .pword 0x7504
  .pword 0x74F0
  .pword 0x74DB
  .pword 0x74C6
  .pword 0x74B2
  .pword 0x749D
  .pword 0x7488
  .pword 0x7474
  .pword 0x745F
  .pword 0x744A
  .pword 0x7435
  .pword 0x7420
  .pword 0x740A
  .pword 0x73F5
  .pword 0x73E0
  .pword 0x73CA
  .pword 0x73B5
  .pword 0x739F
  .pword 0x738A
  .pword 0x7374
  .pword 0x735E
  .pword 0x7349
  .pword 0x7333
  .pword 0x731D
  .pword 0x7307
  .pword 0x72F1
  .pword 0x72DB
  .pword 0x72C4
  .pword 0x72AE
  .pword 0x7298
  .pword 0x7281
  .pword 0x726B
  .pword 0x7254
  .pword 0x723E
  .pword 0x7227
  .pword 0x7210
  .pword 0x71F9
  .pword 0x71E2
  .pword 0x71CB
  .pword 0x71B4
  .pword 0x719D
  .pword 0x7186
  .pword 0x716F
  .pword 0x7158
  .pword 0x7140
  .pword 0x7129
  .pword 0x7111
  .pword 0x70FA
  .pword 0x70E2
  .pword 0x70CA
  .pword 0x70B2
  .pword 0x709B
  .pword 0x7083
  .pword 0x706B
  .pword 0x7053
  .pword 0x703A
  .pword 0x7022
  .pword 0x700A
  .pword 0x6FF2
  .pword 0x6FD9
  .pword 0x6FC1
  .pword 0x6FA8
  .pword 0x6F90
  .pword 0x6F77
  .pword 0x6F5E
  .pword 0x6F45
  .pword 0x6F2C
  .pword 0x6F14
  .pword 0x6EFB
  .pword 0x6EE1
  .pword 0x6EC8
  .pword 0x6EAF
  .pword 0x6E96
  .pword 0x6E7C
  .pword 0x6E63
  .pword 0x6E4A
  .pword 0x6E30
  .pword 0x6E16
  .pword 0x6DFD
  .pword 0x6DE3
  .pword 0x6DC9
  .pword 0x6DAF
  .pword 0x6D95
  .pword 0x6D7B
  .pword 0x6D61
  .pword 0x6D47
  .pword 0x6D2D
  .pword 0x6D13
  .pword 0x6CF8
  .pword 0x6CDE
  .pword 0x6CC3
  .pword 0x6CA9
  .pword 0x6C8E
  .pword 0x6C74
  .pword 0x6C59
  .pword 0x6C3E
  .pword 0x6C23
  .pword 0x6C08
  .pword 0x6BED
  .pword 0x6BD2
  .pword 0x6BB7
  .pword 0x6B9C
  .pword 0x6B81
  .pword 0x6B65
  .pword 0x6B4A
  .pword 0x6B2F
  .pword 0x6B13
  .pword 0x6AF8
  .pword 0x6ADC
  .pword 0x6AC0
  .pword 0x6AA4
  .pword 0x6A89
  .pword 0x6A6D
  .pword 0x6A51
  .pword 0x6A35
  .pword 0x6A19
  .pword 0x69FD
  .pword 0x69E0
  .pword 0x69C4
  .pword 0x69A8
  .pword 0x698B
  .pword 0x696F
  .pword 0x6952
  .pword 0x6936
  .pword 0x6919
  .pword 0x68FC
  .pword 0x68E0
  .pword 0x68C3
  .pword 0x68A6
  .pword 0x6889
  .pword 0x686C
  .pword 0x684F
  .pword 0x6832
  .pword 0x6814
  .pword 0x67F7
  .pword 0x67DA
  .pword 0x67BC
  .pword 0x679F
  .pword 0x6781
  .pword 0x6764
  .pword 0x6746
  .pword 0x6728
  .pword 0x670A
  .pword 0x66ED
  .pword 0x66CF
  .pword 0x66B1
  .pword 0x6693
  .pword 0x6675
  .pword 0x6656
  .pword 0x6638
  .pword 0x661A
  .pword 0x65FC
  .pword 0x65DD
  .pword 0x65BF
  .pword 0x65A0
  .pword 0x6582
  .pword 0x6563
  .pword 0x6544
  .pword 0x6525
  .pword 0x6507
  .pword 0x64E8
  .pword 0x64C9
  .pword 0x64AA
  .pword 0x648B
  .pword 0x646C
  .pword 0x644C
  .pword 0x642D
  .pword 0x640E
  .pword 0x63EE
  .pword 0x63CF
  .pword 0x63AF
  .pword 0x6390
  .pword 0x6370
  .pword 0x6351
  .pword 0x6331
  .pword 0x6311
  .pword 0x62F1
  .pword 0x62D1
  .pword 0x62B1
  .pword 0x6291
  .pword 0x6271
  .pword 0x6251
  .pword 0x6231
  .pword 0x6211
  .pword 0x61F0
  .pword 0x61D0
  .pword 0x61AF
  .pword 0x618F
  .pword 0x616E
  .pword 0x614E
  .pword 0x612D
  .pword 0x610C
  .pword 0x60EB
  .pword 0x60CB
  .pword 0x60AA
  .pword 0x6089
  .pword 0x6068
  .pword 0x6047
  .pword 0x6025
  .pword 0x6004
  .pword 0x5FE3
  .pword 0x5FC2
  .pword 0x5FA0
  .pword 0x5F7F
  .pword 0x5F5D
  .pword 0x5F3C
  .pword 0x5F1A
  .pword 0x5EF8
  .pword 0x5ED7
  .pword 0x5EB5
  .pword 0x5E93
  .pword 0x5E71
  .pword 0x5E4F
  .pword 0x5E2D
  .pword 0x5E0B
  .pword 0x5DE9
  .pword 0x5DC7
  .pword 0x5DA5
  .pword 0x5D82
  .pword 0x5D60
  .pword 0x5D3E
  .pword 0x5D1B
  .pword 0x5CF9
  .pword 0x5CD6
  .pword 0x5CB3
  .pword 0x5C91
  .pword 0x5C6E
  .pword 0x5C4B
  .pword 0x5C28
  .pword 0x5C05
  .pword 0x5BE2
  .pword 0x5BBF
  .pword 0x5B9C
  .pword 0x5B79
  .pword 0x5B56
  .pword 0x5B33
  .pword 0x5B0F
  .pword 0x5AEC
  .pword 0x5AC9
  .pword 0x5AA5
  .pword 0x5A82
  .pword 0x5A5E
  .pword 0x5A3B
  .pword 0x5A17
  .pword 0x59F3
  .pword 0x59CF
  .pword 0x59AC
  .pword 0x5988
  .pword 0x5964
  .pword 0x5940
  .pword 0x591C
  .pword 0x58F8
  .pword 0x58D3
  .pword 0x58AF
  .pword 0x588B
  .pword 0x5867
  .pword 0x5842
  .pword 0x581E
  .pword 0x57F9
  .pword 0x57D5
  .pword 0x57B0
  .pword 0x578B
  .pword 0x5767
  .pword 0x5742
  .pword 0x571D
  .pword 0x56F8
  .pword 0x56D3
  .pword 0x56AF
  .pword 0x568A
  .pword 0x5664
  .pword 0x563F
  .pword 0x561A
  .pword 0x55F5
  .pword 0x55D0
  .pword 0x55AA
  .pword 0x5585
  .pword 0x5560
  .pword 0x553A
  .pword 0x5515
  .pword 0x54EF
  .pword 0x54C9
  .pword 0x54A4
  .pword 0x547E
  .pword 0x5458
  .pword 0x5432
  .pword 0x540C
  .pword 0x53E7
  .pword 0x53C1
  .pword 0x539B
  .pword 0x5374
  .pword 0x534E
  .pword 0x5328
  .pword 0x5302
  .pword 0x52DC
  .pword 0x52B5
  .pword 0x528F
  .pword 0x5268
  .pword 0x5242
  .pword 0x521B
  .pword 0x51F5
  .pword 0x51CE
  .pword 0x51A8
  .pword 0x5181
  .pword 0x515A
  .pword 0x5133
  .pword 0x510C
  .pword 0x50E5
  .pword 0x50BE
  .pword 0x5097
  .pword 0x5070
  .pword 0x5049
  .pword 0x5022
  .pword 0x4FFB
  .pword 0x4FD4
  .pword 0x4FAC
  .pword 0x4F85
  .pword 0x4F5D
  .pword 0x4F36
  .pword 0x4F0E
  .pword 0x4EE7
  .pword 0x4EBF
  .pword 0x4E98
  .pword 0x4E70
  .pword 0x4E48
  .pword 0x4E20
  .pword 0x4DF9
  .pword 0x4DD1
  .pword 0x4DA9
  .pword 0x4D81
  .pword 0x4D59
  .pword 0x4D31
  .pword 0x4D09
  .pword 0x4CE0
  .pword 0x4CB8
  .pword 0x4C90
  .pword 0x4C68
  .pword 0x4C3F
  .pword 0x4C17
  .pword 0x4BEE
  .pword 0x4BC6
  .pword 0x4B9D
  .pword 0x4B75
  .pword 0x4B4C
  .pword 0x4B24
  .pword 0x4AFB
  .pword 0x4AD2
  .pword 0x4AA9
  .pword 0x4A80
  .pword 0x4A58
  .pword 0x4A2F
  .pword 0x4A06
  .pword 0x49DD
  .pword 0x49B4
  .pword 0x498A
  .pword 0x4961
  .pword 0x4938
  .pword 0x490F
  .pword 0x48E5
  .pword 0x48BC
  .pword 0x4893
  .pword 0x4869
  .pword 0x4840
  .pword 0x4816
  .pword 0x47ED
  .pword 0x47C3
  .pword 0x479A
  .pword 0x4770
  .pword 0x4746
  .pword 0x471C
  .pword 0x46F3
  .pword 0x46C9
  .pword 0x469F
  .pword 0x4675
  .pword 0x464B
  .pword 0x4621
  .pword 0x45F7
  .pword 0x45CD
  .pword 0x45A3
  .pword 0x4578
  .pword 0x454E
  .pword 0x4524
  .pword 0x44F9
  .pword 0x44CF
  .pword 0x44A5
  .pword 0x447A
  .pword 0x4450
  .pword 0x4425
  .pword 0x43FB
  .pword 0x43D0
  .pword 0x43A5
  .pword 0x437B
  .pword 0x4350
  .pword 0x4325
  .pword 0x42FA
  .pword 0x42D0
  .pword 0x42A5
  .pword 0x427A
  .pword 0x424F
  .pword 0x4224
  .pword 0x41F9
  .pword 0x41CE
  .pword 0x41A2
  .pword 0x4177
  .pword 0x414C
  .pword 0x4121
  .pword 0x40F6
  .pword 0x40CA
  .pword 0x409F
  .pword 0x4073
  .pword 0x4048
  .pword 0x401D
  .pword 0x3FF1
  .pword 0x3FC5
  .pword 0x3F9A
  .pword 0x3F6E
  .pword 0x3F43
  .pword 0x3F17
  .pword 0x3EEB
  .pword 0x3EBF
  .pword 0x3E93
  .pword 0x3E68
  .pword 0x3E3C
  .pword 0x3E10
  .pword 0x3DE4
  .pword 0x3DB8
  .pword 0x3D8C
  .pword 0x3D60
  .pword 0x3D33
  .pword 0x3D07
  .pword 0x3CDB
  .pword 0x3CAF
  .pword 0x3C83
  .pword 0x3C56
  .pword 0x3C2A
  .pword 0x3BFE
  .pword 0x3BD1
  .pword 0x3BA5
  .pword 0x3B78
  .pword 0x3B4C
  .pword 0x3B1F
  .pword 0x3AF2
  .pword 0x3AC6
  .pword 0x3A99
  .pword 0x3A6C
  .pword 0x3A40
  .pword 0x3A13
  .pword 0x39E6
  .pword 0x39B9
  .pword 0x398C
  .pword 0x3960
  .pword 0x3933
  .pword 0x3906
  .pword 0x38D9
  .pword 0x38AB
  .pword 0x387E
  .pword 0x3851
  .pword 0x3824
  .pword 0x37F7
  .pword 0x37CA
  .pword 0x379C
  .pword 0x376F
  .pword 0x3742
  .pword 0x3715
  .pword 0x36E7
  .pword 0x36BA
  .pword 0x368C
  .pword 0x365F
  .pword 0x3631
  .pword 0x3604
  .pword 0x35D6
  .pword 0x35A8
  .pword 0x357B
  .pword 0x354D
  .pword 0x351F
  .pword 0x34F2
  .pword 0x34C4
  .pword 0x3496
  .pword 0x3468
  .pword 0x343A
  .pword 0x340C
  .pword 0x33DF
  .pword 0x33B1
  .pword 0x3383
  .pword 0x3355
  .pword 0x3326
  .pword 0x32F8
  .pword 0x32CA
  .pword 0x329C
  .pword 0x326E
  .pword 0x3240
  .pword 0x3211
  .pword 0x31E3
  .pword 0x31B5
  .pword 0x3187
  .pword 0x3158
  .pword 0x312A
  .pword 0x30FB
  .pword 0x30CD
  .pword 0x309E
  .pword 0x3070
  .pword 0x3041
  .pword 0x3013
  .pword 0x2FE4
  .pword 0x2FB6
  .pword 0x2F87
  .pword 0x2F58
  .pword 0x2F2A
  .pword 0x2EFB
  .pword 0x2ECC
  .pword 0x2E9D
  .pword 0x2E6E
  .pword 0x2E40
  .pword 0x2E11
  .pword 0x2DE2
  .pword 0x2DB3
  .pword 0x2D84
  .pword 0x2D55
  .pword 0x2D26
  .pword 0x2CF7
  .pword 0x2CC8
  .pword 0x2C99
  .pword 0x2C6A
  .pword 0x2C3A
  .pword 0x2C0B
  .pword 0x2BDC
  .pword 0x2BAD
  .pword 0x2B7D
  .pword 0x2B4E
  .pword 0x2B1F
  .pword 0x2AF0
  .pword 0x2AC0
  .pword 0x2A91
  .pword 0x2A61
  .pword 0x2A32
  .pword 0x2A02
  .pword 0x29D3
  .pword 0x29A3
  .pword 0x2974
  .pword 0x2944
  .pword 0x2915
  .pword 0x28E5
  .pword 0x28B5
  .pword 0x2886
  .pword 0x2856
  .pword 0x2826
  .pword 0x27F7
  .pword 0x27C7
  .pword 0x2797
  .pword 0x2767
  .pword 0x2737
  .pword 0x2708
  .pword 0x26D8
  .pword 0x26A8
  .pword 0x2678
  .pword 0x2648
  .pword 0x2618
  .pword 0x25E8
  .pword 0x25B8
  .pword 0x2588
  .pword 0x2558
  .pword 0x2528
  .pword 0x24F8
  .pword 0x24C8
  .pword 0x2497
  .pword 0x2467
  .pword 0x2437
  .pword 0x2407
  .pword 0x23D7
  .pword 0x23A6
  .pword 0x2376
  .pword 0x2346
  .pword 0x2315
  .pword 0x22E5
  .pword 0x22B5
  .pword 0x2284
  .pword 0x2254
  .pword 0x2223
  .pword 0x21F3
  .pword 0x21C2
  .pword 0x2192
  .pword 0x2161
  .pword 0x2131
  .pword 0x2100
  .pword 0x20D0
  .pword 0x209F
  .pword 0x206F
  .pword 0x203E
  .pword 0x200D
  .pword 0x1FDD
  .pword 0x1FAC
  .pword 0x1F7B
  .pword 0x1F4A
  .pword 0x1F1A
  .pword 0x1EE9
  .pword 0x1EB8
  .pword 0x1E87
  .pword 0x1E57
  .pword 0x1E26
  .pword 0x1DF5
  .pword 0x1DC4
  .pword 0x1D93
  .pword 0x1D62
  .pword 0x1D31
  .pword 0x1D00
  .pword 0x1CCF
  .pword 0x1C9E
  .pword 0x1C6D
  .pword 0x1C3C
  .pword 0x1C0B
  .pword 0x1BDA
  .pword 0x1BA9
  .pword 0x1B78
  .pword 0x1B47
  .pword 0x1B16
  .pword 0x1AE5
  .pword 0x1AB4
  .pword 0x1A82
  .pword 0x1A51
  .pword 0x1A20
  .pword 0x19EF
  .pword 0x19BE
  .pword 0x198C
  .pword 0x195B
  .pword 0x192A
  .pword 0x18F9
  .pword 0x18C7
  .pword 0x1896
  .pword 0x1865
  .pword 0x1833
  .pword 0x1802
  .pword 0x17D0
  .pword 0x179F
  .pword 0x176E
  .pword 0x173C
  .pword 0x170B
  .pword 0x16D9
  .pword 0x16A8
  .pword 0x1676
  .pword 0x1645
  .pword 0x1613
  .pword 0x15E2
  .pword 0x15B0
  .pword 0x157F
  .pword 0x154D
  .pword 0x151C
  .pword 0x14EA
  .pword 0x14B9
  .pword 0x1487
  .pword 0x1455
  .pword 0x1424
  .pword 0x13F2
  .pword 0x13C0
  .pword 0x138F
  .pword 0x135D
  .pword 0x132B
  .pword 0x12FA
  .pword 0x12C8
  .pword 0x1296
  .pword 0x1264
  .pword 0x1233
  .pword 0x1201
  .pword 0x11CF
  .pword 0x119D
  .pword 0x116C
  .pword 0x113A
  .pword 0x1108
  .pword 0x10D6
  .pword 0x10A4
  .pword 0x1072
  .pword 0x1041
  .pword 0x100F
  .pword 0x0FDD
  .pword 0x0FAB
  .pword 0x0F79
  .pword 0x0F47
  .pword 0x0F15
  .pword 0x0EE3
  .pword 0x0EB1
  .pword 0x0E80
  .pword 0x0E4E
  .pword 0x0E1C
  .pword 0x0DEA
  .pword 0x0DB8
  .pword 0x0D86
  .pword 0x0D54
  .pword 0x0D22
  .pword 0x0CF0
  .pword 0x0CBE
  .pword 0x0C8C
  .pword 0x0C5A
  .pword 0x0C28
  .pword 0x0BF6
  .pword 0x0BC4
  .pword 0x0B92
  .pword 0x0B5F
  .pword 0x0B2D
  .pword 0x0AFB
  .pword 0x0AC9
  .pword 0x0A97
  .pword 0x0A65
  .pword 0x0A33
  .pword 0x0A01
  .pword 0x09CF
  .pword 0x099D
  .pword 0x096A
  .pword 0x0938
  .pword 0x0906
  .pword 0x08D4
  .pword 0x08A2
  .pword 0x0870
  .pword 0x083E
  .pword 0x080B
  .pword 0x07D9
  .pword 0x07A7
  .pword 0x0775
  .pword 0x0743
  .pword 0x0711
  .pword 0x06DE
  .pword 0x06AC
  .pword 0x067A
  .pword 0x0648
  .pword 0x0616
  .pword 0x05E3
  .pword 0x05B1
  .pword 0x057F
  .pword 0x054D
  .pword 0x051B
  .pword 0x04E8
  .pword 0x04B6
  .pword 0x0484
  .pword 0x0452
  .pword 0x041F
  .pword 0x03ED
  .pword 0x03BB
  .pword 0x0389
  .pword 0x0356
  .pword 0x0324
  .pword 0x02F2
  .pword 0x02C0
  .pword 0x028D
  .pword 0x025B
  .pword 0x0229
  .pword 0x01F7
  .pword 0x01C4
  .pword 0x0192
  .pword 0x0160
  .pword 0x012E
  .pword 0x00FB
  .pword 0x00C9
  .pword 0x0097
  .pword 0x0065
  .pword 0x0032
  .pword 0x0000
  .pword 0xFFCF
  .pword 0xFF9C
  .pword 0xFF6A
  .pword 0xFF38
  .pword 0xFF06
  .pword 0xFED3
  .pword 0xFEA1
  .pword 0xFE6F
  .pword 0xFE3D
  .pword 0xFE0A
  .pword 0xFDD8
  .pword 0xFDA6
  .pword 0xFD74
  .pword 0xFD41
  .pword 0xFD0F
  .pword 0xFCDD
  .pword 0xFCAB
  .pword 0xFC78
  .pword 0xFC46
  .pword 0xFC14
  .pword 0xFBE2
  .pword 0xFBAF
  .pword 0xFB7D
  .pword 0xFB4B
  .pword 0xFB19
  .pword 0xFAE6
  .pword 0xFAB4
  .pword 0xFA82
  .pword 0xFA50
  .pword 0xFA1E
  .pword 0xF9EB
  .pword 0xF9B9
  .pword 0xF987
  .pword 0xF955
  .pword 0xF923
  .pword 0xF8F0
  .pword 0xF8BE
  .pword 0xF88C
  .pword 0xF85A
  .pword 0xF828
  .pword 0xF7F6
  .pword 0xF7C3
  .pword 0xF791
  .pword 0xF75F
  .pword 0xF72D
  .pword 0xF6FB
  .pword 0xF6C9
  .pword 0xF697
  .pword 0xF664
  .pword 0xF632
  .pword 0xF600
  .pword 0xF5CE
  .pword 0xF59C
  .pword 0xF56A
  .pword 0xF538
  .pword 0xF506
  .pword 0xF4D4
  .pword 0xF4A2
  .pword 0xF46F
  .pword 0xF43D
  .pword 0xF40B
  .pword 0xF3D9
  .pword 0xF3A7
  .pword 0xF375
  .pword 0xF343
  .pword 0xF311
  .pword 0xF2DF
  .pword 0xF2AD
  .pword 0xF27B
  .pword 0xF249
  .pword 0xF217
  .pword 0xF1E5
  .pword 0xF1B3
  .pword 0xF181
  .pword 0xF150
  .pword 0xF11E
  .pword 0xF0EC
  .pword 0xF0BA
  .pword 0xF088
  .pword 0xF056
  .pword 0xF024
  .pword 0xEFF2
  .pword 0xEFC0
  .pword 0xEF8F
  .pword 0xEF5D
  .pword 0xEF2B
  .pword 0xEEF9
  .pword 0xEEC7
  .pword 0xEE95
  .pword 0xEE64
  .pword 0xEE32
  .pword 0xEE00
  .pword 0xEDCE
  .pword 0xED9D
  .pword 0xED6B
  .pword 0xED39
  .pword 0xED07
  .pword 0xECD6
  .pword 0xECA4
  .pword 0xEC72
  .pword 0xEC41
  .pword 0xEC0F
  .pword 0xEBDD
  .pword 0xEBAC
  .pword 0xEB7A
  .pword 0xEB48
  .pword 0xEB17
  .pword 0xEAE5
  .pword 0xEAB4
  .pword 0xEA82
  .pword 0xEA51
  .pword 0xEA1F
  .pword 0xE9EE
  .pword 0xE9BC
  .pword 0xE98B
  .pword 0xE959
  .pword 0xE928
  .pword 0xE8F6
  .pword 0xE8C5
  .pword 0xE893
  .pword 0xE862
  .pword 0xE831
  .pword 0xE7FF
  .pword 0xE7CE
  .pword 0xE79C
  .pword 0xE76B
  .pword 0xE73A
  .pword 0xE708
  .pword 0xE6D7
  .pword 0xE6A6
  .pword 0xE675
  .pword 0xE643
  .pword 0xE612
  .pword 0xE5E1
  .pword 0xE5B0
  .pword 0xE57F
  .pword 0xE54D
  .pword 0xE51C
  .pword 0xE4EB
  .pword 0xE4BA
  .pword 0xE489
  .pword 0xE458
  .pword 0xE427
  .pword 0xE3F6
  .pword 0xE3C5
  .pword 0xE394
  .pword 0xE363
  .pword 0xE332
  .pword 0xE301
  .pword 0xE2D0
  .pword 0xE29F
  .pword 0xE26E
  .pword 0xE23D
  .pword 0xE20C
  .pword 0xE1DB
  .pword 0xE1AA
  .pword 0xE17A
  .pword 0xE149
  .pword 0xE118
  .pword 0xE0E7
  .pword 0xE0B7
  .pword 0xE086
  .pword 0xE055
  .pword 0xE024
  .pword 0xDFF4
  .pword 0xDFC3
  .pword 0xDF92
  .pword 0xDF62
  .pword 0xDF31
  .pword 0xDF01
  .pword 0xDED0
  .pword 0xDEA0
  .pword 0xDE6F
  .pword 0xDE3F
  .pword 0xDE0E
  .pword 0xDDDE
  .pword 0xDDAD
  .pword 0xDD7D
  .pword 0xDD4C
  .pword 0xDD1C
  .pword 0xDCEC
  .pword 0xDCBB
  .pword 0xDC8B
  .pword 0xDC5B
  .pword 0xDC2A
  .pword 0xDBFA
  .pword 0xDBCA
  .pword 0xDB9A
  .pword 0xDB6A
  .pword 0xDB39
  .pword 0xDB09
  .pword 0xDAD9
  .pword 0xDAA9
  .pword 0xDA79
  .pword 0xDA49
  .pword 0xDA19
  .pword 0xD9E9
  .pword 0xD9B9
  .pword 0xD989
  .pword 0xD959
  .pword 0xD929
  .pword 0xD8F9
  .pword 0xD8CA
  .pword 0xD89A
  .pword 0xD86A
  .pword 0xD83A
  .pword 0xD80A
  .pword 0xD7DB
  .pword 0xD7AB
  .pword 0xD77B
  .pword 0xD74C
  .pword 0xD71C
  .pword 0xD6EC
  .pword 0xD6BD
  .pword 0xD68D
  .pword 0xD65E
  .pword 0xD62E
  .pword 0xD5FF
  .pword 0xD5CF
  .pword 0xD5A0
  .pword 0xD570
  .pword 0xD541
  .pword 0xD511
  .pword 0xD4E2
  .pword 0xD4B3
  .pword 0xD484
  .pword 0xD454
  .pword 0xD425
  .pword 0xD3F6
  .pword 0xD3C7
  .pword 0xD397
  .pword 0xD368
  .pword 0xD339
  .pword 0xD30A
  .pword 0xD2DB
  .pword 0xD2AC
  .pword 0xD27D
  .pword 0xD24E
  .pword 0xD21F
  .pword 0xD1F0
  .pword 0xD1C1
  .pword 0xD193
  .pword 0xD164
  .pword 0xD135
  .pword 0xD106
  .pword 0xD0D7
  .pword 0xD0A9
  .pword 0xD07A
  .pword 0xD04B
  .pword 0xD01D
  .pword 0xCFEE
  .pword 0xCFC0
  .pword 0xCF91
  .pword 0xCF63
  .pword 0xCF34
  .pword 0xCF06
  .pword 0xCED7
  .pword 0xCEA9
  .pword 0xCE7A
  .pword 0xCE4C
  .pword 0xCE1E
  .pword 0xCDF0
  .pword 0xCDC1
  .pword 0xCD93
  .pword 0xCD65
  .pword 0xCD37
  .pword 0xCD09
  .pword 0xCCDB
  .pword 0xCCAC
  .pword 0xCC7E
  .pword 0xCC50
  .pword 0xCC22
  .pword 0xCBF5
  .pword 0xCBC7
  .pword 0xCB99
  .pword 0xCB6B
  .pword 0xCB3D
  .pword 0xCB0F
  .pword 0xCAE2
  .pword 0xCAB4
  .pword 0xCA86
  .pword 0xCA59
  .pword 0xCA2B
  .pword 0xC9FD
  .pword 0xC9D0
  .pword 0xC9A2
  .pword 0xC975
  .pword 0xC947
  .pword 0xC91A
  .pword 0xC8EC
  .pword 0xC8BF
  .pword 0xC892
  .pword 0xC865
  .pword 0xC837
  .pword 0xC80A
  .pword 0xC7DD
  .pword 0xC7B0
  .pword 0xC783
  .pword 0xC756
  .pword 0xC728
  .pword 0xC6FB
  .pword 0xC6CE
  .pword 0xC6A1
  .pword 0xC675
  .pword 0xC648
  .pword 0xC61B
  .pword 0xC5EE
  .pword 0xC5C1
  .pword 0xC595
  .pword 0xC568
  .pword 0xC53B
  .pword 0xC50F
  .pword 0xC4E2
  .pword 0xC4B5
  .pword 0xC489
  .pword 0xC45C
  .pword 0xC430
  .pword 0xC403
  .pword 0xC3D7
  .pword 0xC3AB
  .pword 0xC37E
  .pword 0xC352
  .pword 0xC326
  .pword 0xC2FA
  .pword 0xC2CE
  .pword 0xC2A1
  .pword 0xC275
  .pword 0xC249
  .pword 0xC21D
  .pword 0xC1F1
  .pword 0xC1C5
  .pword 0xC199
  .pword 0xC16E
  .pword 0xC142
  .pword 0xC116
  .pword 0xC0EA
  .pword 0xC0BE
  .pword 0xC093
  .pword 0xC067
  .pword 0xC03C
  .pword 0xC010
  .pword 0xBFE4
  .pword 0xBFB9
  .pword 0xBF8E
  .pword 0xBF62
  .pword 0xBF37
  .pword 0xBF0B
  .pword 0xBEE0
  .pword 0xBEB5
  .pword 0xBE8A
  .pword 0xBE5F
  .pword 0xBE33
  .pword 0xBE08
  .pword 0xBDDD
  .pword 0xBDB2
  .pword 0xBD87
  .pword 0xBD5C
  .pword 0xBD31
  .pword 0xBD07
  .pword 0xBCDC
  .pword 0xBCB1
  .pword 0xBC86
  .pword 0xBC5C
  .pword 0xBC31
  .pword 0xBC06
  .pword 0xBBDC
  .pword 0xBBB1
  .pword 0xBB87
  .pword 0xBB5C
  .pword 0xBB32
  .pword 0xBB08
  .pword 0xBADD
  .pword 0xBAB3
  .pword 0xBA89
  .pword 0xBA5E
  .pword 0xBA34
  .pword 0xBA0A
  .pword 0xB9E0
  .pword 0xB9B6
  .pword 0xB98C
  .pword 0xB962
  .pword 0xB938
  .pword 0xB90E
  .pword 0xB8E5
  .pword 0xB8BB
  .pword 0xB891
  .pword 0xB867
  .pword 0xB83E
  .pword 0xB814
  .pword 0xB7EB
  .pword 0xB7C1
  .pword 0xB798
  .pword 0xB76E
  .pword 0xB745
  .pword 0xB71C
  .pword 0xB6F2
  .pword 0xB6C9
  .pword 0xB6A0
  .pword 0xB677
  .pword 0xB64D
  .pword 0xB624
  .pword 0xB5FB
  .pword 0xB5D2
  .pword 0xB5A9
  .pword 0xB581
  .pword 0xB558
  .pword 0xB52F
  .pword 0xB506
  .pword 0xB4DD
  .pword 0xB4B5
  .pword 0xB48C
  .pword 0xB464
  .pword 0xB43B
  .pword 0xB413
  .pword 0xB3EA
  .pword 0xB3C2
  .pword 0xB399
  .pword 0xB371
  .pword 0xB349
  .pword 0xB321
  .pword 0xB2F8
  .pword 0xB2D0
  .pword 0xB2A8
  .pword 0xB280
  .pword 0xB258
  .pword 0xB230
  .pword 0xB208
  .pword 0xB1E1
  .pword 0xB1B9
  .pword 0xB191
  .pword 0xB169
  .pword 0xB142
  .pword 0xB11A
  .pword 0xB0F3
  .pword 0xB0CB
  .pword 0xB0A4
  .pword 0xB07C
  .pword 0xB055
  .pword 0xB02D
  .pword 0xB006
  .pword 0xAFDF
  .pword 0xAFB8
  .pword 0xAF91
  .pword 0xAF6A
  .pword 0xAF43
  .pword 0xAF1C
  .pword 0xAEF5
  .pword 0xAECE
  .pword 0xAEA7
  .pword 0xAE80
  .pword 0xAE59
  .pword 0xAE33
  .pword 0xAE0C
  .pword 0xADE6
  .pword 0xADBF
  .pword 0xAD99
  .pword 0xAD72
  .pword 0xAD4C
  .pword 0xAD25
  .pword 0xACFF
  .pword 0xACD9
  .pword 0xACB3
  .pword 0xAC8D
  .pword 0xAC66
  .pword 0xAC40
  .pword 0xAC1A
  .pword 0xABF5
  .pword 0xABCF
  .pword 0xABA9
  .pword 0xAB83
  .pword 0xAB5D
  .pword 0xAB38
  .pword 0xAB12
  .pword 0xAAEC
  .pword 0xAAC7
  .pword 0xAAA1
  .pword 0xAA7C
  .pword 0xAA57
  .pword 0xAA31
  .pword 0xAA0C
  .pword 0xA9E7
  .pword 0xA9C2
  .pword 0xA99D
  .pword 0xA977
  .pword 0xA952
  .pword 0xA92E
  .pword 0xA909
  .pword 0xA8E4
  .pword 0xA8BF
  .pword 0xA89A
  .pword 0xA876
  .pword 0xA851
  .pword 0xA82C
  .pword 0xA808
  .pword 0xA7E3
  .pword 0xA7BF
  .pword 0xA79A
  .pword 0xA776
  .pword 0xA752
  .pword 0xA72E
  .pword 0xA709
  .pword 0xA6E5
  .pword 0xA6C1
  .pword 0xA69D
  .pword 0xA679
  .pword 0xA655
  .pword 0xA632
  .pword 0xA60E
  .pword 0xA5EA
  .pword 0xA5C6
  .pword 0xA5A3
  .pword 0xA57F
  .pword 0xA55C
  .pword 0xA538
  .pword 0xA515
  .pword 0xA4F2
  .pword 0xA4CE
  .pword 0xA4AB
  .pword 0xA488
  .pword 0xA465
  .pword 0xA442
  .pword 0xA41F
  .pword 0xA3FC
  .pword 0xA3D9
  .pword 0xA3B6
  .pword 0xA393
  .pword 0xA370
  .pword 0xA34E
  .pword 0xA32B
  .pword 0xA308
  .pword 0xA2E6
  .pword 0xA2C3
  .pword 0xA2A1
  .pword 0xA27F
  .pword 0xA25C
  .pword 0xA23A
  .pword 0xA218
  .pword 0xA1F6
  .pword 0xA1D4
  .pword 0xA1B2
  .pword 0xA190
  .pword 0xA16E
  .pword 0xA14C
  .pword 0xA12A
  .pword 0xA109
  .pword 0xA0E7
  .pword 0xA0C5
  .pword 0xA0A4
  .pword 0xA082
  .pword 0xA061
  .pword 0xA03F
  .pword 0xA01E
  .pword 0x9FFD
  .pword 0x9FDC
  .pword 0x9FBA
  .pword 0x9F99
  .pword 0x9F78
  .pword 0x9F57
  .pword 0x9F36
  .pword 0x9F16
  .pword 0x9EF5
  .pword 0x9ED4
  .pword 0x9EB3
  .pword 0x9E93
  .pword 0x9E72
  .pword 0x9E52
  .pword 0x9E31
  .pword 0x9E11
  .pword 0x9DF0
  .pword 0x9DD0
  .pword 0x9DB0
  .pword 0x9D90
  .pword 0x9D70
  .pword 0x9D50
  .pword 0x9D30
  .pword 0x9D10
  .pword 0x9CF0
  .pword 0x9CD0
  .pword 0x9CB0
  .pword 0x9C91
  .pword 0x9C71
  .pword 0x9C52
  .pword 0x9C32
  .pword 0x9C13
  .pword 0x9BF3
  .pword 0x9BD4
  .pword 0x9BB5
  .pword 0x9B95
  .pword 0x9B76
  .pword 0x9B57
  .pword 0x9B38
  .pword 0x9B19
  .pword 0x9AFA
  .pword 0x9ADC
  .pword 0x9ABD
  .pword 0x9A9E
  .pword 0x9A7F
  .pword 0x9A61
  .pword 0x9A42
  .pword 0x9A24
  .pword 0x9A05
  .pword 0x99E7
  .pword 0x99C9
  .pword 0x99AB
  .pword 0x998C
  .pword 0x996E
  .pword 0x9950
  .pword 0x9932
  .pword 0x9914
  .pword 0x98F7
  .pword 0x98D9
  .pword 0x98BB
  .pword 0x989D
  .pword 0x9880
  .pword 0x9862
  .pword 0x9845
  .pword 0x9827
  .pword 0x980A
  .pword 0x97ED
  .pword 0x97CF
  .pword 0x97B2
  .pword 0x9795
  .pword 0x9778
  .pword 0x975B
  .pword 0x973E
  .pword 0x9721
  .pword 0x9705
  .pword 0x96E8
  .pword 0x96CB
  .pword 0x96AF
  .pword 0x9692
  .pword 0x9676
  .pword 0x9659
  .pword 0x963D
  .pword 0x9621
  .pword 0x9604
  .pword 0x95E8
  .pword 0x95CC
  .pword 0x95B0
  .pword 0x9594
  .pword 0x9578
  .pword 0x955D
  .pword 0x9541
  .pword 0x9525
  .pword 0x9509
  .pword 0x94EE
  .pword 0x94D2
  .pword 0x94B7
  .pword 0x949C
  .pword 0x9480
  .pword 0x9465
  .pword 0x944A
  .pword 0x942F
  .pword 0x9414
  .pword 0x93F9
  .pword 0x93DE
  .pword 0x93C3
  .pword 0x93A8
  .pword 0x938D
  .pword 0x9373
  .pword 0x9358
  .pword 0x933E
  .pword 0x9323
  .pword 0x9309
  .pword 0x92EE
  .pword 0x92D4
  .pword 0x92BA
  .pword 0x92A0
  .pword 0x9286
  .pword 0x926C
  .pword 0x9252
  .pword 0x9238
  .pword 0x921E
  .pword 0x9204
  .pword 0x91EB
  .pword 0x91D1
  .pword 0x91B7
  .pword 0x919E
  .pword 0x9185
  .pword 0x916B
  .pword 0x9152
  .pword 0x9139
  .pword 0x9120
  .pword 0x9106
  .pword 0x90ED
  .pword 0x90D5
  .pword 0x90BC
  .pword 0x90A3
  .pword 0x908A
  .pword 0x9071
  .pword 0x9059
  .pword 0x9040
  .pword 0x9028
  .pword 0x900F
  .pword 0x8FF7
  .pword 0x8FDF
  .pword 0x8FC7
  .pword 0x8FAE
  .pword 0x8F96
  .pword 0x8F7E
  .pword 0x8F66
  .pword 0x8F4F
  .pword 0x8F37
  .pword 0x8F1F
  .pword 0x8F07
  .pword 0x8EF0
  .pword 0x8ED8
  .pword 0x8EC1
  .pword 0x8EA9
  .pword 0x8E92
  .pword 0x8E7B
  .pword 0x8E64
  .pword 0x8E4D
  .pword 0x8E36
  .pword 0x8E1F
  .pword 0x8E08
  .pword 0x8DF1
  .pword 0x8DDA
  .pword 0x8DC3
  .pword 0x8DAD
  .pword 0x8D96
  .pword 0x8D80
  .pword 0x8D69
  .pword 0x8D53
  .pword 0x8D3D
  .pword 0x8D26
  .pword 0x8D10
  .pword 0x8CFA
  .pword 0x8CE4
  .pword 0x8CCE
  .pword 0x8CB8
  .pword 0x8CA3
  .pword 0x8C8D
  .pword 0x8C77
  .pword 0x8C62
  .pword 0x8C4C
  .pword 0x8C37
  .pword 0x8C21
  .pword 0x8C0C
  .pword 0x8BF7
  .pword 0x8BE1
  .pword 0x8BCC
  .pword 0x8BB7
  .pword 0x8BA2
  .pword 0x8B8D
  .pword 0x8B79
  .pword 0x8B64
  .pword 0x8B4F
  .pword 0x8B3B
  .pword 0x8B26
  .pword 0x8B11
  .pword 0x8AFD
  .pword 0x8AE9
  .pword 0x8AD4
  .pword 0x8AC0
  .pword 0x8AAC
  .pword 0x8A98
  .pword 0x8A84
  .pword 0x8A70
  .pword 0x8A5C
  .pword 0x8A49
  .pword 0x8A35
  .pword 0x8A21
  .pword 0x8A0E
  .pword 0x89FA
  .pword 0x89E7
  .pword 0x89D4
  .pword 0x89C0
  .pword 0x89AD
  .pword 0x899A
  .pword 0x8987
  .pword 0x8974
  .pword 0x8961
  .pword 0x894E
  .pword 0x893B
  .pword 0x8929
  .pword 0x8916
  .pword 0x8903
  .pword 0x88F1
  .pword 0x88DF
  .pword 0x88CC
  .pword 0x88BA
  .pword 0x88A8
  .pword 0x8896
  .pword 0x8884
  .pword 0x8872
  .pword 0x8860
  .pword 0x884E
  .pword 0x883C
  .pword 0x882A
  .pword 0x8819
  .pword 0x8807
  .pword 0x87F6
  .pword 0x87E4
  .pword 0x87D3
  .pword 0x87C2
  .pword 0x87B1
  .pword 0x879F
  .pword 0x878E
  .pword 0x877D
  .pword 0x876D
  .pword 0x875C
  .pword 0x874B
  .pword 0x873A
  .pword 0x872A
  .pword 0x8719
  .pword 0x8709
  .pword 0x86F8
  .pword 0x86E8
  .pword 0x86D8
  .pword 0x86C8
  .pword 0x86B7
  .pword 0x86A7
  .pword 0x8697
  .pword 0x8688
  .pword 0x8678
  .pword 0x8668
  .pword 0x8658
  .pword 0x8649
  .pword 0x8639
  .pword 0x862A
  .pword 0x861B
  .pword 0x860B
  .pword 0x85FC
  .pword 0x85ED
  .pword 0x85DE
  .pword 0x85CF
  .pword 0x85C0
  .pword 0x85B1
  .pword 0x85A2
  .pword 0x8594
  .pword 0x8585
  .pword 0x8576
  .pword 0x8568
  .pword 0x8559
  .pword 0x854B
  .pword 0x853D
  .pword 0x852F
  .pword 0x8521
  .pword 0x8513
  .pword 0x8505
  .pword 0x84F7
  .pword 0x84E9
  .pword 0x84DB
  .pword 0x84CE
  .pword 0x84C0
  .pword 0x84B2
  .pword 0x84A5
  .pword 0x8498
  .pword 0x848A
  .pword 0x847D
  .pword 0x8470
  .pword 0x8463
  .pword 0x8456
  .pword 0x8449
  .pword 0x843C
  .pword 0x842F
  .pword 0x8423
  .pword 0x8416
  .pword 0x8409
  .pword 0x83FD
  .pword 0x83F1
  .pword 0x83E4
  .pword 0x83D8
  .pword 0x83CC
  .pword 0x83C0
  .pword 0x83B4
  .pword 0x83A8
  .pword 0x839C
  .pword 0x8390
  .pword 0x8384
  .pword 0x8379
  .pword 0x836D
  .pword 0x8362
  .pword 0x8356
  .pword 0x834B
  .pword 0x8340
  .pword 0x8334
  .pword 0x8329
  .pword 0x831E
  .pword 0x8313
  .pword 0x8308
  .pword 0x82FD
  .pword 0x82F3
  .pword 0x82E8
  .pword 0x82DD
  .pword 0x82D3
  .pword 0x82C8
  .pword 0x82BE
  .pword 0x82B4
  .pword 0x82AA
  .pword 0x829F
  .pword 0x8295
  .pword 0x828B
  .pword 0x8281
  .pword 0x8278
  .pword 0x826E
  .pword 0x8264
  .pword 0x825B
  .pword 0x8251
  .pword 0x8248
  .pword 0x823E
  .pword 0x8235
  .pword 0x822C
  .pword 0x8222
  .pword 0x8219
  .pword 0x8210
  .pword 0x8207
  .pword 0x81FF
  .pword 0x81F6
  .pword 0x81ED
  .pword 0x81E4
  .pword 0x81DC
  .pword 0x81D3
  .pword 0x81CB
  .pword 0x81C3
  .pword 0x81BA
  .pword 0x81B2
  .pword 0x81AA
  .pword 0x81A2
  .pword 0x819A
  .pword 0x8192
  .pword 0x818A
  .pword 0x8183
  .pword 0x817B
  .pword 0x8174
  .pword 0x816C
  .pword 0x8165
  .pword 0x815D
  .pword 0x8156
  .pword 0x814F
  .pword 0x8148
  .pword 0x8141
  .pword 0x813A
  .pword 0x8133
  .pword 0x812C
  .pword 0x8125
  .pword 0x811F
  .pword 0x8118
  .pword 0x8112
  .pword 0x810B
  .pword 0x8105
  .pword 0x80FF
  .pword 0x80F8
  .pword 0x80F2
  .pword 0x80EC
  .pword 0x80E6
  .pword 0x80E0
  .pword 0x80DB
  .pword 0x80D5
  .pword 0x80CF
  .pword 0x80CA
  .pword 0x80C4
  .pword 0x80BF
  .pword 0x80B9
  .pword 0x80B4
  .pword 0x80AF
  .pword 0x80AA
  .pword 0x80A5
  .pword 0x80A0
  .pword 0x809B
  .pword 0x8096
  .pword 0x8091
  .pword 0x808D
  .pword 0x8088
  .pword 0x8084
  .pword 0x807F
  .pword 0x807B
  .pword 0x8077
  .pword 0x8072
  .pword 0x806E
  .pword 0x806A
  .pword 0x8066
  .pword 0x8062
  .pword 0x805F
  .pword 0x805B
  .pword 0x8057
  .pword 0x8054
  .pword 0x8050
  .pword 0x804D
  .pword 0x8049
  .pword 0x8046
  .pword 0x8043
  .pword 0x8040
  .pword 0x803D
  .pword 0x803A
  .pword 0x8037
  .pword 0x8034
  .pword 0x8031
  .pword 0x802F
  .pword 0x802C
  .pword 0x8029
  .pword 0x8027
  .pword 0x8025
  .pword 0x8022
  .pword 0x8020
  .pword 0x801E
  .pword 0x801C
  .pword 0x801A
  .pword 0x8018
  .pword 0x8016
  .pword 0x8015
  .pword 0x8013
  .pword 0x8011
  .pword 0x8010
  .pword 0x800E
  .pword 0x800D
  .pword 0x800C
  .pword 0x800B
  .pword 0x800A
  .pword 0x8009
  .pword 0x8008
  .pword 0x8007
  .pword 0x8006
  .pword 0x8005
  .pword 0x8004
  .pword 0x8004
  .pword 0x8003
  .pword 0x8003
  .pword 0x8003
  .pword 0x8002
  .pword 0x8002
  .pword 0x8002
  .pword 0x8002
  .pword 0x8002
  .pword 0x8002
  .pword 0x8002
  .pword 0x8003
  .pword 0x8003
  .pword 0x8003
  .pword 0x8004
  .pword 0x8004
  .pword 0x8005
  .pword 0x8006
  .pword 0x8007
  .pword 0x8008
  .pword 0x8009
  .pword 0x800A
  .pword 0x800B
  .pword 0x800C
  .pword 0x800D
  .pword 0x800E
  .pword 0x8010
  .pword 0x8011
  .pword 0x8013
  .pword 0x8015
  .pword 0x8016
  .pword 0x8018
  .pword 0x801A
  .pword 0x801C
  .pword 0x801E
  .pword 0x8020
  .pword 0x8022
  .pword 0x8025
  .pword 0x8027
  .pword 0x8029
  .pword 0x802C
  .pword 0x802F
  .pword 0x8031
  .pword 0x8034
  .pword 0x8037
  .pword 0x803A
  .pword 0x803D
  .pword 0x8040
  .pword 0x8043
  .pword 0x8046
  .pword 0x8049
  .pword 0x804D
  .pword 0x8050
  .pword 0x8054
  .pword 0x8057
  .pword 0x805B
  .pword 0x805F
  .pword 0x8062
  .pword 0x8066
  .pword 0x806A
  .pword 0x806E
  .pword 0x8072
  .pword 0x8077
  .pword 0x807B
  .pword 0x807F
  .pword 0x8084
  .pword 0x8088
  .pword 0x808D
  .pword 0x8091
  .pword 0x8096
  .pword 0x809B
  .pword 0x80A0
  .pword 0x80A5
  .pword 0x80AA
  .pword 0x80AF
  .pword 0x80B4
  .pword 0x80B9
  .pword 0x80BF
  .pword 0x80C4
  .pword 0x80CA
  .pword 0x80CF
  .pword 0x80D5
  .pword 0x80DB
  .pword 0x80E0
  .pword 0x80E6
  .pword 0x80EC
  .pword 0x80F2
  .pword 0x80F8
  .pword 0x80FF
  .pword 0x8105
  .pword 0x810B
  .pword 0x8112
  .pword 0x8118
  .pword 0x811F
  .pword 0x8125
  .pword 0x812C
  .pword 0x8133
  .pword 0x813A
  .pword 0x8141
  .pword 0x8148
  .pword 0x814F
  .pword 0x8156
  .pword 0x815D
  .pword 0x8165
  .pword 0x816C
  .pword 0x8174
  .pword 0x817B
  .pword 0x8183
  .pword 0x818A
  .pword 0x8192
  .pword 0x819A
  .pword 0x81A2
  .pword 0x81AA
  .pword 0x81B2
  .pword 0x81BA
  .pword 0x81C3
  .pword 0x81CB
  .pword 0x81D3
  .pword 0x81DC
  .pword 0x81E4
  .pword 0x81ED
  .pword 0x81F6
  .pword 0x81FF
  .pword 0x8207
  .pword 0x8210
  .pword 0x8219
  .pword 0x8222
  .pword 0x822C
  .pword 0x8235
  .pword 0x823E
  .pword 0x8248
  .pword 0x8251
  .pword 0x825B
  .pword 0x8264
  .pword 0x826E
  .pword 0x8278
  .pword 0x8281
  .pword 0x828B
  .pword 0x8295
  .pword 0x829F
  .pword 0x82AA
  .pword 0x82B4
  .pword 0x82BE
  .pword 0x82C8
  .pword 0x82D3
  .pword 0x82DD
  .pword 0x82E8
  .pword 0x82F3
  .pword 0x82FD
  .pword 0x8308
  .pword 0x8313
  .pword 0x831E
  .pword 0x8329
  .pword 0x8334
  .pword 0x8340
  .pword 0x834B
  .pword 0x8356
  .pword 0x8362
  .pword 0x836D
  .pword 0x8379
  .pword 0x8384
  .pword 0x8390
  .pword 0x839C
  .pword 0x83A8
  .pword 0x83B4
  .pword 0x83C0
  .pword 0x83CC
  .pword 0x83D8
  .pword 0x83E4
  .pword 0x83F1
  .pword 0x83FD
  .pword 0x8409
  .pword 0x8416
  .pword 0x8423
  .pword 0x842F
  .pword 0x843C
  .pword 0x8449
  .pword 0x8456
  .pword 0x8463
  .pword 0x8470
  .pword 0x847D
  .pword 0x848A
  .pword 0x8498
  .pword 0x84A5
  .pword 0x84B2
  .pword 0x84C0
  .pword 0x84CE
  .pword 0x84DB
  .pword 0x84E9
  .pword 0x84F7
  .pword 0x8505
  .pword 0x8513
  .pword 0x8521
  .pword 0x852F
  .pword 0x853D
  .pword 0x854B
  .pword 0x8559
  .pword 0x8568
  .pword 0x8576
  .pword 0x8585
  .pword 0x8594
  .pword 0x85A2
  .pword 0x85B1
  .pword 0x85C0
  .pword 0x85CF
  .pword 0x85DE
  .pword 0x85ED
  .pword 0x85FC
  .pword 0x860B
  .pword 0x861B
  .pword 0x862A
  .pword 0x8639
  .pword 0x8649
  .pword 0x8658
  .pword 0x8668
  .pword 0x8678
  .pword 0x8688
  .pword 0x8697
  .pword 0x86A7
  .pword 0x86B7
  .pword 0x86C8
  .pword 0x86D8
  .pword 0x86E8
  .pword 0x86F8
  .pword 0x8709
  .pword 0x8719
  .pword 0x872A
  .pword 0x873A
  .pword 0x874B
  .pword 0x875C
  .pword 0x876D
  .pword 0x877D
  .pword 0x878E
  .pword 0x879F
  .pword 0x87B1
  .pword 0x87C2
  .pword 0x87D3
  .pword 0x87E4
  .pword 0x87F6
  .pword 0x8807
  .pword 0x8819
  .pword 0x882A
  .pword 0x883C
  .pword 0x884E
  .pword 0x8860
  .pword 0x8872
  .pword 0x8884
  .pword 0x8896
  .pword 0x88A8
  .pword 0x88BA
  .pword 0x88CC
  .pword 0x88DF
  .pword 0x88F1
  .pword 0x8903
  .pword 0x8916
  .pword 0x8929
  .pword 0x893B
  .pword 0x894E
  .pword 0x8961
  .pword 0x8974
  .pword 0x8987
  .pword 0x899A
  .pword 0x89AD
  .pword 0x89C0
  .pword 0x89D4
  .pword 0x89E7
  .pword 0x89FA
  .pword 0x8A0E
  .pword 0x8A21
  .pword 0x8A35
  .pword 0x8A49
  .pword 0x8A5C
  .pword 0x8A70
  .pword 0x8A84
  .pword 0x8A98
  .pword 0x8AAC
  .pword 0x8AC0
  .pword 0x8AD4
  .pword 0x8AE9
  .pword 0x8AFD
  .pword 0x8B11
  .pword 0x8B26
  .pword 0x8B3B
  .pword 0x8B4F
  .pword 0x8B64
  .pword 0x8B79
  .pword 0x8B8D
  .pword 0x8BA2
  .pword 0x8BB7
  .pword 0x8BCC
  .pword 0x8BE1
  .pword 0x8BF7
  .pword 0x8C0C
  .pword 0x8C21
  .pword 0x8C37
  .pword 0x8C4C
  .pword 0x8C62
  .pword 0x8C77
  .pword 0x8C8D
  .pword 0x8CA3
  .pword 0x8CB8
  .pword 0x8CCE
  .pword 0x8CE4
  .pword 0x8CFA
  .pword 0x8D10
  .pword 0x8D26
  .pword 0x8D3D
  .pword 0x8D53
  .pword 0x8D69
  .pword 0x8D80
  .pword 0x8D96
  .pword 0x8DAD
  .pword 0x8DC3
  .pword 0x8DDA
  .pword 0x8DF1
  .pword 0x8E08
  .pword 0x8E1F
  .pword 0x8E36
  .pword 0x8E4D
  .pword 0x8E64
  .pword 0x8E7B
  .pword 0x8E92
  .pword 0x8EA9
  .pword 0x8EC1
  .pword 0x8ED8
  .pword 0x8EF0
  .pword 0x8F07
  .pword 0x8F1F
  .pword 0x8F37
  .pword 0x8F4F
  .pword 0x8F66
  .pword 0x8F7E
  .pword 0x8F96
  .pword 0x8FAE
  .pword 0x8FC7
  .pword 0x8FDF
  .pword 0x8FF7
  .pword 0x900F
  .pword 0x9028
  .pword 0x9040
  .pword 0x9059
  .pword 0x9071
  .pword 0x908A
  .pword 0x90A3
  .pword 0x90BC
  .pword 0x90D5
  .pword 0x90ED
  .pword 0x9106
  .pword 0x9120
  .pword 0x9139
  .pword 0x9152
  .pword 0x916B
  .pword 0x9185
  .pword 0x919E
  .pword 0x91B7
  .pword 0x91D1
  .pword 0x91EB
  .pword 0x9204
  .pword 0x921E
  .pword 0x9238
  .pword 0x9252
  .pword 0x926C
  .pword 0x9286
  .pword 0x92A0
  .pword 0x92BA
  .pword 0x92D4
  .pword 0x92EE
  .pword 0x9309
  .pword 0x9323
  .pword 0x933E
  .pword 0x9358
  .pword 0x9373
  .pword 0x938D
  .pword 0x93A8
  .pword 0x93C3
  .pword 0x93DE
  .pword 0x93F9
  .pword 0x9414
  .pword 0x942F
  .pword 0x944A
  .pword 0x9465
  .pword 0x9480
  .pword 0x949C
  .pword 0x94B7
  .pword 0x94D2
  .pword 0x94EE
  .pword 0x9509
  .pword 0x9525
  .pword 0x9541
  .pword 0x955D
  .pword 0x9578
  .pword 0x9594
  .pword 0x95B0
  .pword 0x95CC
  .pword 0x95E8
  .pword 0x9604
  .pword 0x9621
  .pword 0x963D
  .pword 0x9659
  .pword 0x9676
  .pword 0x9692
  .pword 0x96AF
  .pword 0x96CB
  .pword 0x96E8
  .pword 0x9705
  .pword 0x9721
  .pword 0x973E
  .pword 0x975B
  .pword 0x9778
  .pword 0x9795
  .pword 0x97B2
  .pword 0x97CF
  .pword 0x97ED
  .pword 0x980A
  .pword 0x9827
  .pword 0x9845
  .pword 0x9862
  .pword 0x9880
  .pword 0x989D
  .pword 0x98BB
  .pword 0x98D9
  .pword 0x98F7
  .pword 0x9914
  .pword 0x9932
  .pword 0x9950
  .pword 0x996E
  .pword 0x998C
  .pword 0x99AB
  .pword 0x99C9
  .pword 0x99E7
  .pword 0x9A05
  .pword 0x9A24
  .pword 0x9A42
  .pword 0x9A61
  .pword 0x9A7F
  .pword 0x9A9E
  .pword 0x9ABD
  .pword 0x9ADC
  .pword 0x9AFA
  .pword 0x9B19
  .pword 0x9B38
  .pword 0x9B57
  .pword 0x9B76
  .pword 0x9B95
  .pword 0x9BB5
  .pword 0x9BD4
  .pword 0x9BF3
  .pword 0x9C13
  .pword 0x9C32
  .pword 0x9C52
  .pword 0x9C71
  .pword 0x9C91
  .pword 0x9CB0
  .pword 0x9CD0
  .pword 0x9CF0
  .pword 0x9D10
  .pword 0x9D30
  .pword 0x9D50
  .pword 0x9D70
  .pword 0x9D90
  .pword 0x9DB0
  .pword 0x9DD0
  .pword 0x9DF0
  .pword 0x9E11
  .pword 0x9E31
  .pword 0x9E52
  .pword 0x9E72
  .pword 0x9E93
  .pword 0x9EB3
  .pword 0x9ED4
  .pword 0x9EF5
  .pword 0x9F16
  .pword 0x9F36
  .pword 0x9F57
  .pword 0x9F78
  .pword 0x9F99
  .pword 0x9FBA
  .pword 0x9FDC
  .pword 0x9FFD
  .pword 0xA01E
  .pword 0xA03F
  .pword 0xA061
  .pword 0xA082
  .pword 0xA0A4
  .pword 0xA0C5
  .pword 0xA0E7
  .pword 0xA109
  .pword 0xA12A
  .pword 0xA14C
  .pword 0xA16E
  .pword 0xA190
  .pword 0xA1B2
  .pword 0xA1D4
  .pword 0xA1F6
  .pword 0xA218
  .pword 0xA23A
  .pword 0xA25C
  .pword 0xA27F
  .pword 0xA2A1
  .pword 0xA2C3
  .pword 0xA2E6
  .pword 0xA308
  .pword 0xA32B
  .pword 0xA34E
  .pword 0xA370
  .pword 0xA393
  .pword 0xA3B6
  .pword 0xA3D9
  .pword 0xA3FC
  .pword 0xA41F
  .pword 0xA442
  .pword 0xA465
  .pword 0xA488
  .pword 0xA4AB
  .pword 0xA4CE
  .pword 0xA4F2
  .pword 0xA515
  .pword 0xA538
  .pword 0xA55C
  .pword 0xA57F
  .pword 0xA5A3
  .pword 0xA5C6
  .pword 0xA5EA
  .pword 0xA60E
  .pword 0xA632
  .pword 0xA655
  .pword 0xA679
  .pword 0xA69D
  .pword 0xA6C1
  .pword 0xA6E5
  .pword 0xA709
  .pword 0xA72E
  .pword 0xA752
  .pword 0xA776
  .pword 0xA79A
  .pword 0xA7BF
  .pword 0xA7E3
  .pword 0xA808
  .pword 0xA82C
  .pword 0xA851
  .pword 0xA876
  .pword 0xA89A
  .pword 0xA8BF
  .pword 0xA8E4
  .pword 0xA909
  .pword 0xA92E
  .pword 0xA952
  .pword 0xA977
  .pword 0xA99D
  .pword 0xA9C2
  .pword 0xA9E7
  .pword 0xAA0C
  .pword 0xAA31
  .pword 0xAA57
  .pword 0xAA7C
  .pword 0xAAA1
  .pword 0xAAC7
  .pword 0xAAEC
  .pword 0xAB12
  .pword 0xAB38
  .pword 0xAB5D
  .pword 0xAB83
  .pword 0xABA9
  .pword 0xABCF
  .pword 0xABF5
  .pword 0xAC1A
  .pword 0xAC40
  .pword 0xAC66
  .pword 0xAC8D
  .pword 0xACB3
  .pword 0xACD9
  .pword 0xACFF
  .pword 0xAD25
  .pword 0xAD4C
  .pword 0xAD72
  .pword 0xAD99
  .pword 0xADBF
  .pword 0xADE6
  .pword 0xAE0C
  .pword 0xAE33
  .pword 0xAE59
  .pword 0xAE80
  .pword 0xAEA7
  .pword 0xAECE
  .pword 0xAEF5
  .pword 0xAF1C
  .pword 0xAF43
  .pword 0xAF6A
  .pword 0xAF91
  .pword 0xAFB8
  .pword 0xAFDF
  .pword 0xB006
  .pword 0xB02D
  .pword 0xB055
  .pword 0xB07C
  .pword 0xB0A4
  .pword 0xB0CB
  .pword 0xB0F3
  .pword 0xB11A
  .pword 0xB142
  .pword 0xB169
  .pword 0xB191
  .pword 0xB1B9
  .pword 0xB1E1
  .pword 0xB208
  .pword 0xB230
  .pword 0xB258
  .pword 0xB280
  .pword 0xB2A8
  .pword 0xB2D0
  .pword 0xB2F8
  .pword 0xB321
  .pword 0xB349
  .pword 0xB371
  .pword 0xB399
  .pword 0xB3C2
  .pword 0xB3EA
  .pword 0xB413
  .pword 0xB43B
  .pword 0xB464
  .pword 0xB48C
  .pword 0xB4B5
  .pword 0xB4DD
  .pword 0xB506
  .pword 0xB52F
  .pword 0xB558
  .pword 0xB581
  .pword 0xB5A9
  .pword 0xB5D2
  .pword 0xB5FB
  .pword 0xB624
  .pword 0xB64D
  .pword 0xB677
  .pword 0xB6A0
  .pword 0xB6C9
  .pword 0xB6F2
  .pword 0xB71C
  .pword 0xB745
  .pword 0xB76E
  .pword 0xB798
  .pword 0xB7C1
  .pword 0xB7EB
  .pword 0xB814
  .pword 0xB83E
  .pword 0xB867
  .pword 0xB891
  .pword 0xB8BB
  .pword 0xB8E5
  .pword 0xB90E
  .pword 0xB938
  .pword 0xB962
  .pword 0xB98C
  .pword 0xB9B6
  .pword 0xB9E0
  .pword 0xBA0A
  .pword 0xBA34
  .pword 0xBA5E
  .pword 0xBA89
  .pword 0xBAB3
  .pword 0xBADD
  .pword 0xBB08
  .pword 0xBB32
  .pword 0xBB5C
  .pword 0xBB87
  .pword 0xBBB1
  .pword 0xBBDC
  .pword 0xBC06
  .pword 0xBC31
  .pword 0xBC5C
  .pword 0xBC86
  .pword 0xBCB1
  .pword 0xBCDC
  .pword 0xBD07
  .pword 0xBD31
  .pword 0xBD5C
  .pword 0xBD87
  .pword 0xBDB2
  .pword 0xBDDD
  .pword 0xBE08
  .pword 0xBE33
  .pword 0xBE5F
  .pword 0xBE8A
  .pword 0xBEB5
  .pword 0xBEE0
  .pword 0xBF0B
  .pword 0xBF37
  .pword 0xBF62
  .pword 0xBF8E
  .pword 0xBFB9
  .pword 0xBFE4
  .pword 0xC010
  .pword 0xC03C
  .pword 0xC067
  .pword 0xC093
  .pword 0xC0BE
  .pword 0xC0EA
  .pword 0xC116
  .pword 0xC142
  .pword 0xC16E
  .pword 0xC199
  .pword 0xC1C5
  .pword 0xC1F1
  .pword 0xC21D
  .pword 0xC249
  .pword 0xC275
  .pword 0xC2A1
  .pword 0xC2CE
  .pword 0xC2FA
  .pword 0xC326
  .pword 0xC352
  .pword 0xC37E
  .pword 0xC3AB
  .pword 0xC3D7
  .pword 0xC403
  .pword 0xC430
  .pword 0xC45C
  .pword 0xC489
  .pword 0xC4B5
  .pword 0xC4E2
  .pword 0xC50F
  .pword 0xC53B
  .pword 0xC568
  .pword 0xC595
  .pword 0xC5C1
  .pword 0xC5EE
  .pword 0xC61B
  .pword 0xC648
  .pword 0xC675
  .pword 0xC6A1
  .pword 0xC6CE
  .pword 0xC6FB
  .pword 0xC728
  .pword 0xC756
  .pword 0xC783
  .pword 0xC7B0
  .pword 0xC7DD
  .pword 0xC80A
  .pword 0xC837
  .pword 0xC865
  .pword 0xC892
  .pword 0xC8BF
  .pword 0xC8EC
  .pword 0xC91A
  .pword 0xC947
  .pword 0xC975
  .pword 0xC9A2
  .pword 0xC9D0
  .pword 0xC9FD
  .pword 0xCA2B
  .pword 0xCA59
  .pword 0xCA86
  .pword 0xCAB4
  .pword 0xCAE2
  .pword 0xCB0F
  .pword 0xCB3D
  .pword 0xCB6B
  .pword 0xCB99
  .pword 0xCBC7
  .pword 0xCBF5
  .pword 0xCC22
  .pword 0xCC50
  .pword 0xCC7E
  .pword 0xCCAC
  .pword 0xCCDB
  .pword 0xCD09
  .pword 0xCD37
  .pword 0xCD65
  .pword 0xCD93
  .pword 0xCDC1
  .pword 0xCDF0
  .pword 0xCE1E
  .pword 0xCE4C
  .pword 0xCE7A
  .pword 0xCEA9
  .pword 0xCED7
  .pword 0xCF06
  .pword 0xCF34
  .pword 0xCF63
  .pword 0xCF91
  .pword 0xCFC0
  .pword 0xCFEE
  .pword 0xD01D
  .pword 0xD04B
  .pword 0xD07A
  .pword 0xD0A9
  .pword 0xD0D7
  .pword 0xD106
  .pword 0xD135
  .pword 0xD164
  .pword 0xD193
  .pword 0xD1C1
  .pword 0xD1F0
  .pword 0xD21F
  .pword 0xD24E
  .pword 0xD27D
  .pword 0xD2AC
  .pword 0xD2DB
  .pword 0xD30A
  .pword 0xD339
  .pword 0xD368
  .pword 0xD397
  .pword 0xD3C7
  .pword 0xD3F6
  .pword 0xD425
  .pword 0xD454
  .pword 0xD484
  .pword 0xD4B3
  .pword 0xD4E2
  .pword 0xD511
  .pword 0xD541
  .pword 0xD570
  .pword 0xD5A0
  .pword 0xD5CF
  .pword 0xD5FF
  .pword 0xD62E
  .pword 0xD65E
  .pword 0xD68D
  .pword 0xD6BD
  .pword 0xD6EC
  .pword 0xD71C
  .pword 0xD74C
  .pword 0xD77B
  .pword 0xD7AB
  .pword 0xD7DB
  .pword 0xD80A
  .pword 0xD83A
  .pword 0xD86A
  .pword 0xD89A
  .pword 0xD8CA
  .pword 0xD8F9
  .pword 0xD929
  .pword 0xD959
  .pword 0xD989
  .pword 0xD9B9
  .pword 0xD9E9
  .pword 0xDA19
  .pword 0xDA49
  .pword 0xDA79
  .pword 0xDAA9
  .pword 0xDAD9
  .pword 0xDB09
  .pword 0xDB39
  .pword 0xDB6A
  .pword 0xDB9A
  .pword 0xDBCA
  .pword 0xDBFA
  .pword 0xDC2A
  .pword 0xDC5B
  .pword 0xDC8B
  .pword 0xDCBB
  .pword 0xDCEC
  .pword 0xDD1C
  .pword 0xDD4C
  .pword 0xDD7D
  .pword 0xDDAD
  .pword 0xDDDE
  .pword 0xDE0E
  .pword 0xDE3F
  .pword 0xDE6F
  .pword 0xDEA0
  .pword 0xDED0
  .pword 0xDF01
  .pword 0xDF31
  .pword 0xDF62
  .pword 0xDF92
  .pword 0xDFC3
  .pword 0xDFF4
  .pword 0xE024
  .pword 0xE055
  .pword 0xE086
  .pword 0xE0B7
  .pword 0xE0E7
  .pword 0xE118
  .pword 0xE149
  .pword 0xE17A
  .pword 0xE1AA
  .pword 0xE1DB
  .pword 0xE20C
  .pword 0xE23D
  .pword 0xE26E
  .pword 0xE29F
  .pword 0xE2D0
  .pword 0xE301
  .pword 0xE332
  .pword 0xE363
  .pword 0xE394
  .pword 0xE3C5
  .pword 0xE3F6
  .pword 0xE427
  .pword 0xE458
  .pword 0xE489
  .pword 0xE4BA
  .pword 0xE4EB
  .pword 0xE51C
  .pword 0xE54D
  .pword 0xE57F
  .pword 0xE5B0
  .pword 0xE5E1
  .pword 0xE612
  .pword 0xE643
  .pword 0xE675
  .pword 0xE6A6
  .pword 0xE6D7
  .pword 0xE708
  .pword 0xE73A
  .pword 0xE76B
  .pword 0xE79C
  .pword 0xE7CE
  .pword 0xE7FF
  .pword 0xE831
  .pword 0xE862
  .pword 0xE893
  .pword 0xE8C5
  .pword 0xE8F6
  .pword 0xE928
  .pword 0xE959
  .pword 0xE98B
  .pword 0xE9BC
  .pword 0xE9EE
  .pword 0xEA1F
  .pword 0xEA51
  .pword 0xEA82
  .pword 0xEAB4
  .pword 0xEAE5
  .pword 0xEB17
  .pword 0xEB48
  .pword 0xEB7A
  .pword 0xEBAC
  .pword 0xEBDD
  .pword 0xEC0F
  .pword 0xEC41
  .pword 0xEC72
  .pword 0xECA4
  .pword 0xECD6
  .pword 0xED07
  .pword 0xED39
  .pword 0xED6B
  .pword 0xED9D
  .pword 0xEDCE
  .pword 0xEE00
  .pword 0xEE32
  .pword 0xEE64
  .pword 0xEE95
  .pword 0xEEC7
  .pword 0xEEF9
  .pword 0xEF2B
  .pword 0xEF5D
  .pword 0xEF8F
  .pword 0xEFC0
  .pword 0xEFF2
  .pword 0xF024
  .pword 0xF056
  .pword 0xF088
  .pword 0xF0BA
  .pword 0xF0EC
  .pword 0xF11E
  .pword 0xF150
  .pword 0xF181
  .pword 0xF1B3
  .pword 0xF1E5
  .pword 0xF217
  .pword 0xF249
  .pword 0xF27B
  .pword 0xF2AD
  .pword 0xF2DF
  .pword 0xF311
  .pword 0xF343
  .pword 0xF375
  .pword 0xF3A7
  .pword 0xF3D9
  .pword 0xF40B
  .pword 0xF43D
  .pword 0xF46F
  .pword 0xF4A2
  .pword 0xF4D4
  .pword 0xF506
  .pword 0xF538
  .pword 0xF56A
  .pword 0xF59C
  .pword 0xF5CE
  .pword 0xF600
  .pword 0xF632
  .pword 0xF664
  .pword 0xF697
  .pword 0xF6C9
  .pword 0xF6FB
  .pword 0xF72D
  .pword 0xF75F
  .pword 0xF791
  .pword 0xF7C3
  .pword 0xF7F6
  .pword 0xF828
  .pword 0xF85A
  .pword 0xF88C
  .pword 0xF8BE
  .pword 0xF8F0
  .pword 0xF923
  .pword 0xF955
  .pword 0xF987
  .pword 0xF9B9
  .pword 0xF9EB
  .pword 0xFA1E
  .pword 0xFA50
  .pword 0xFA82
  .pword 0xFAB4
  .pword 0xFAE6
  .pword 0xFB19
  .pword 0xFB4B
  .pword 0xFB7D
  .pword 0xFBAF
  .pword 0xFBE2
  .pword 0xFC14
  .pword 0xFC46
  .pword 0xFC78
  .pword 0xFCAB
  .pword 0xFCDD
  .pword 0xFD0F
  .pword 0xFD41
  .pword 0xFD74
  .pword 0xFDA6
  .pword 0xFDD8
  .pword 0xFE0A
  .pword 0xFE3D
  .pword 0xFE6F
  .pword 0xFEA1
  .pword 0xFED3
  .pword 0xFF06
  .pword 0xFF38
  .pword 0xFF6A
  .pword 0xFF9C
  .pword 0xFFCF

distorted_sine_table:
  .pword 0x0000
  .pword 0x0064
  .pword 0x00C9
  .pword 0x012D
  .pword 0x0191
  .pword 0x01F5
  .pword 0x0258
  .pword 0x02BC
  .pword 0x031F
  .pword 0x0382
  .pword 0x03E6
  .pword 0x0448
  .pword 0x04AB
  .pword 0x050E
  .pword 0x0570
  .pword 0x05D2
  .pword 0x0635
  .pword 0x0697
  .pword 0x06F8
  .pword 0x075A
  .pword 0x07BB
  .pword 0x081D
  .pword 0x087E
  .pword 0x08DF
  .pword 0x0940
  .pword 0x09A0
  .pword 0x0A01
  .pword 0x0A61
  .pword 0x0AC2
  .pword 0x0B22
  .pword 0x0B81
  .pword 0x0BE1
  .pword 0x0C41
  .pword 0x0CA0
  .pword 0x0CFF
  .pword 0x0D5E
  .pword 0x0DBD
  .pword 0x0E1C
  .pword 0x0E7B
  .pword 0x0ED9
  .pword 0x0F37
  .pword 0x0F95
  .pword 0x0FF3
  .pword 0x1051
  .pword 0x10AF
  .pword 0x110C
  .pword 0x116A
  .pword 0x11C7
  .pword 0x1224
  .pword 0x1280
  .pword 0x12DD
  .pword 0x133A
  .pword 0x1396
  .pword 0x13F2
  .pword 0x144E
  .pword 0x14AA
  .pword 0x1505
  .pword 0x1561
  .pword 0x15BC
  .pword 0x1617
  .pword 0x1672
  .pword 0x16CD
  .pword 0x1728
  .pword 0x1782
  .pword 0x17DD
  .pword 0x1837
  .pword 0x1891
  .pword 0x18EB
  .pword 0x1944
  .pword 0x199E
  .pword 0x19F7
  .pword 0x1A50
  .pword 0x1AA9
  .pword 0x1B02
  .pword 0x1B5B
  .pword 0x1BB3
  .pword 0x1C0B
  .pword 0x1C64
  .pword 0x1CBC
  .pword 0x1D13
  .pword 0x1D6B
  .pword 0x1DC3
  .pword 0x1E1A
  .pword 0x1E71
  .pword 0x1EC8
  .pword 0x1F1F
  .pword 0x1F75
  .pword 0x1FCC
  .pword 0x2022
  .pword 0x2078
  .pword 0x20CE
  .pword 0x2124
  .pword 0x217A
  .pword 0x21CF
  .pword 0x2224
  .pword 0x2279
  .pword 0x22CE
  .pword 0x2323
  .pword 0x2378
  .pword 0x23CC
  .pword 0x2420
  .pword 0x2474
  .pword 0x24C8
  .pword 0x251C
  .pword 0x2570
  .pword 0x25C3
  .pword 0x2616
  .pword 0x2669
  .pword 0x26BC
  .pword 0x270F
  .pword 0x2762
  .pword 0x27B4
  .pword 0x2806
  .pword 0x2858
  .pword 0x28AA
  .pword 0x28FC
  .pword 0x294D
  .pword 0x299F
  .pword 0x29F0
  .pword 0x2A41
  .pword 0x2A92
  .pword 0x2AE2
  .pword 0x2B33
  .pword 0x2B83
  .pword 0x2BD3
  .pword 0x2C23
  .pword 0x2C73
  .pword 0x2CC2
  .pword 0x2D12
  .pword 0x2D61
  .pword 0x2DB0
  .pword 0x2DFF
  .pword 0x2E4E
  .pword 0x2E9D
  .pword 0x2EEB
  .pword 0x2F39
  .pword 0x2F87
  .pword 0x2FD5
  .pword 0x3023
  .pword 0x3070
  .pword 0x30BE
  .pword 0x310B
  .pword 0x3158
  .pword 0x31A5
  .pword 0x31F2
  .pword 0x323E
  .pword 0x328A
  .pword 0x32D7
  .pword 0x3323
  .pword 0x336E
  .pword 0x33BA
  .pword 0x3406
  .pword 0x3451
  .pword 0x349C
  .pword 0x34E7
  .pword 0x3532
  .pword 0x357C
  .pword 0x35C7
  .pword 0x3611
  .pword 0x365B
  .pword 0x36A5
  .pword 0x36EF
  .pword 0x3738
  .pword 0x3782
  .pword 0x37CB
  .pword 0x3814
  .pword 0x385D
  .pword 0x38A5
  .pword 0x38EE
  .pword 0x3936
  .pword 0x397E
  .pword 0x39C6
  .pword 0x3A0E
  .pword 0x3A56
  .pword 0x3A9D
  .pword 0x3AE5
  .pword 0x3B2C
  .pword 0x3B73
  .pword 0x3BBA
  .pword 0x3C00
  .pword 0x3C47
  .pword 0x3C8D
  .pword 0x3CD3
  .pword 0x3D19
  .pword 0x3D5F
  .pword 0x3DA4
  .pword 0x3DEA
  .pword 0x3E2F
  .pword 0x3E74
  .pword 0x3EB9
  .pword 0x3EFE
  .pword 0x3F42
  .pword 0x3F86
  .pword 0x3FCB
  .pword 0x400F
  .pword 0x4052
  .pword 0x4096
  .pword 0x40DA
  .pword 0x411D
  .pword 0x4160
  .pword 0x41A3
  .pword 0x41E6
  .pword 0x4228
  .pword 0x426B
  .pword 0x42AD
  .pword 0x42EF
  .pword 0x4331
  .pword 0x4373
  .pword 0x43B5
  .pword 0x43F6
  .pword 0x4437
  .pword 0x4478
  .pword 0x44B9
  .pword 0x44FA
  .pword 0x453B
  .pword 0x457B
  .pword 0x45BB
  .pword 0x45FB
  .pword 0x463B
  .pword 0x467B
  .pword 0x46BA
  .pword 0x46FA
  .pword 0x4739
  .pword 0x4778
  .pword 0x47B7
  .pword 0x47F6
  .pword 0x4834
  .pword 0x4872
  .pword 0x48B1
  .pword 0x48EF
  .pword 0x492C
  .pword 0x496A
  .pword 0x49A8
  .pword 0x49E5
  .pword 0x4A22
  .pword 0x4A5F
  .pword 0x4A9C
  .pword 0x4AD8
  .pword 0x4B15
  .pword 0x4B51
  .pword 0x4B8D
  .pword 0x4BC9
  .pword 0x4C05
  .pword 0x4C41
  .pword 0x4C7C
  .pword 0x4CB7
  .pword 0x4CF2
  .pword 0x4D2D
  .pword 0x4D68
  .pword 0x4DA3
  .pword 0x4DDD
  .pword 0x4E17
  .pword 0x4E51
  .pword 0x4E8B
  .pword 0x4EC5
  .pword 0x4EFF
  .pword 0x4F38
  .pword 0x4F71
  .pword 0x4FAA
  .pword 0x4FE3
  .pword 0x501C
  .pword 0x5055
  .pword 0x508D
  .pword 0x50C5
  .pword 0x50FD
  .pword 0x5135
  .pword 0x516D
  .pword 0x51A5
  .pword 0x51DC
  .pword 0x5213
  .pword 0x524A
  .pword 0x5281
  .pword 0x52B8
  .pword 0x52EF
  .pword 0x5325
  .pword 0x535B
  .pword 0x5391
  .pword 0x53C7
  .pword 0x53FD
  .pword 0x5433
  .pword 0x5468
  .pword 0x549D
  .pword 0x54D2
  .pword 0x5507
  .pword 0x553C
  .pword 0x5571
  .pword 0x55A5
  .pword 0x55D9
  .pword 0x560D
  .pword 0x5641
  .pword 0x5675
  .pword 0x56A9
  .pword 0x56DC
  .pword 0x5710
  .pword 0x5743
  .pword 0x5776
  .pword 0x57A8
  .pword 0x57DB
  .pword 0x580E
  .pword 0x5840
  .pword 0x5872
  .pword 0x58A4
  .pword 0x58D6
  .pword 0x5907
  .pword 0x5939
  .pword 0x596A
  .pword 0x599B
  .pword 0x59CD
  .pword 0x59FD
  .pword 0x5A2E
  .pword 0x5A5F
  .pword 0x5A8F
  .pword 0x5ABF
  .pword 0x5AEF
  .pword 0x5B1F
  .pword 0x5B4F
  .pword 0x5B7E
  .pword 0x5BAE
  .pword 0x5BDD
  .pword 0x5C0C
  .pword 0x5C3B
  .pword 0x5C6A
  .pword 0x5C99
  .pword 0x5CC7
  .pword 0x5CF5
  .pword 0x5D24
  .pword 0x5D52
  .pword 0x5D7F
  .pword 0x5DAD
  .pword 0x5DDB
  .pword 0x5E08
  .pword 0x5E35
  .pword 0x5E62
  .pword 0x5E8F
  .pword 0x5EBC
  .pword 0x5EE8
  .pword 0x5F15
  .pword 0x5F41
  .pword 0x5F6D
  .pword 0x5F99
  .pword 0x5FC5
  .pword 0x5FF1
  .pword 0x601C
  .pword 0x6048
  .pword 0x6073
  .pword 0x609E
  .pword 0x60C9
  .pword 0x60F3
  .pword 0x611E
  .pword 0x6148
  .pword 0x6173
  .pword 0x619D
  .pword 0x61C7
  .pword 0x61F1
  .pword 0x621A
  .pword 0x6244
  .pword 0x626D
  .pword 0x6296
  .pword 0x62BF
  .pword 0x62E8
  .pword 0x6311
  .pword 0x633A
  .pword 0x6362
  .pword 0x638B
  .pword 0x63B3
  .pword 0x63DB
  .pword 0x6403
  .pword 0x642A
  .pword 0x6452
  .pword 0x6479
  .pword 0x64A1
  .pword 0x64C8
  .pword 0x64EF
  .pword 0x6515
  .pword 0x653C
  .pword 0x6563
  .pword 0x6589
  .pword 0x65AF
  .pword 0x65D5
  .pword 0x65FB
  .pword 0x6621
  .pword 0x6647
  .pword 0x666C
  .pword 0x6692
  .pword 0x66B7
  .pword 0x66DC
  .pword 0x6701
  .pword 0x6726
  .pword 0x674A
  .pword 0x676F
  .pword 0x6793
  .pword 0x67B7
  .pword 0x67DC
  .pword 0x67FF
  .pword 0x6823
  .pword 0x6847
  .pword 0x686A
  .pword 0x688E
  .pword 0x68B1
  .pword 0x68D4
  .pword 0x68F7
  .pword 0x691A
  .pword 0x693C
  .pword 0x695F
  .pword 0x6981
  .pword 0x69A4
  .pword 0x69C6
  .pword 0x69E8
  .pword 0x6A09
  .pword 0x6A2B
  .pword 0x6A4D
  .pword 0x6A6E
  .pword 0x6A8F
  .pword 0x6AB0
  .pword 0x6AD1
  .pword 0x6AF2
  .pword 0x6B13
  .pword 0x6B34
  .pword 0x6B54
  .pword 0x6B74
  .pword 0x6B94
  .pword 0x6BB4
  .pword 0x6BD4
  .pword 0x6BF4
  .pword 0x6C14
  .pword 0x6C33
  .pword 0x6C53
  .pword 0x6C72
  .pword 0x6C91
  .pword 0x6CB0
  .pword 0x6CCF
  .pword 0x6CED
  .pword 0x6D0C
  .pword 0x6D2A
  .pword 0x6D49
  .pword 0x6D67
  .pword 0x6D85
  .pword 0x6DA3
  .pword 0x6DC1
  .pword 0x6DDE
  .pword 0x6DFC
  .pword 0x6E19
  .pword 0x6E36
  .pword 0x6E53
  .pword 0x6E70
  .pword 0x6E8D
  .pword 0x6EAA
  .pword 0x6EC7
  .pword 0x6EE3
  .pword 0x6EFF
  .pword 0x6F1C
  .pword 0x6F38
  .pword 0x6F54
  .pword 0x6F70
  .pword 0x6F8B
  .pword 0x6FA7
  .pword 0x6FC2
  .pword 0x6FDE
  .pword 0x6FF9
  .pword 0x7014
  .pword 0x702F
  .pword 0x704A
  .pword 0x7064
  .pword 0x707F
  .pword 0x7099
  .pword 0x70B4
  .pword 0x70CE
  .pword 0x70E8
  .pword 0x7102
  .pword 0x711C
  .pword 0x7136
  .pword 0x714F
  .pword 0x7169
  .pword 0x7182
  .pword 0x719B
  .pword 0x71B4
  .pword 0x71CD
  .pword 0x71E6
  .pword 0x71FF
  .pword 0x7218
  .pword 0x7230
  .pword 0x7249
  .pword 0x7261
  .pword 0x7279
  .pword 0x7291
  .pword 0x72A9
  .pword 0x72C1
  .pword 0x72D9
  .pword 0x72F0
  .pword 0x7308
  .pword 0x731F
  .pword 0x7336
  .pword 0x734D
  .pword 0x7364
  .pword 0x737B
  .pword 0x7392
  .pword 0x73A9
  .pword 0x73BF
  .pword 0x73D6
  .pword 0x73EC
  .pword 0x7402
  .pword 0x7418
  .pword 0x742E
  .pword 0x7444
  .pword 0x745A
  .pword 0x7470
  .pword 0x7485
  .pword 0x749B
  .pword 0x74B0
  .pword 0x74C5
  .pword 0x74DA
  .pword 0x74EF
  .pword 0x7504
  .pword 0x7519
  .pword 0x752D
  .pword 0x7542
  .pword 0x7556
  .pword 0x756B
  .pword 0x757F
  .pword 0x7593
  .pword 0x75A7
  .pword 0x75BB
  .pword 0x75CF
  .pword 0x75E3
  .pword 0x75F6
  .pword 0x760A
  .pword 0x761D
  .pword 0x7630
  .pword 0x7643
  .pword 0x7656
  .pword 0x7669
  .pword 0x767C
  .pword 0x768F
  .pword 0x76A2
  .pword 0x76B4
  .pword 0x76C7
  .pword 0x76D9
  .pword 0x76EB
  .pword 0x76FD
  .pword 0x770F
  .pword 0x7721
  .pword 0x7733
  .pword 0x7745
  .pword 0x7757
  .pword 0x7768
  .pword 0x777A
  .pword 0x778B
  .pword 0x779C
  .pword 0x77AD
  .pword 0x77BE
  .pword 0x77CF
  .pword 0x77E0
  .pword 0x77F1
  .pword 0x7802
  .pword 0x7812
  .pword 0x7823
  .pword 0x7833
  .pword 0x7843
  .pword 0x7853
  .pword 0x7863
  .pword 0x7873
  .pword 0x7883
  .pword 0x7893
  .pword 0x78A3
  .pword 0x78B2
  .pword 0x78C2
  .pword 0x78D1
  .pword 0x78E1
  .pword 0x78F0
  .pword 0x78FF
  .pword 0x790E
  .pword 0x791D
  .pword 0x792C
  .pword 0x793B
  .pword 0x794A
  .pword 0x7958
  .pword 0x7967
  .pword 0x7975
  .pword 0x7983
  .pword 0x7992
  .pword 0x79A0
  .pword 0x79AE
  .pword 0x79BC
  .pword 0x79CA
  .pword 0x79D8
  .pword 0x79E5
  .pword 0x79F3
  .pword 0x7A01
  .pword 0x7A0E
  .pword 0x7A1B
  .pword 0x7A29
  .pword 0x7A36
  .pword 0x7A43
  .pword 0x7A50
  .pword 0x7A5D
  .pword 0x7A6A
  .pword 0x7A77
  .pword 0x7A84
  .pword 0x7A90
  .pword 0x7A9D
  .pword 0x7AA9
  .pword 0x7AB6
  .pword 0x7AC2
  .pword 0x7ACE
  .pword 0x7ADA
  .pword 0x7AE6
  .pword 0x7AF2
  .pword 0x7AFE
  .pword 0x7B0A
  .pword 0x7B16
  .pword 0x7B21
  .pword 0x7B2D
  .pword 0x7B39
  .pword 0x7B44
  .pword 0x7B4F
  .pword 0x7B5B
  .pword 0x7B66
  .pword 0x7B71
  .pword 0x7B7C
  .pword 0x7B87
  .pword 0x7B92
  .pword 0x7B9D
  .pword 0x7BA7
  .pword 0x7BB2
  .pword 0x7BBD
  .pword 0x7BC7
  .pword 0x7BD2
  .pword 0x7BDC
  .pword 0x7BE6
  .pword 0x7BF1
  .pword 0x7BFB
  .pword 0x7C05
  .pword 0x7C0F
  .pword 0x7C19
  .pword 0x7C23
  .pword 0x7C2C
  .pword 0x7C36
  .pword 0x7C40
  .pword 0x7C49
  .pword 0x7C53
  .pword 0x7C5C
  .pword 0x7C66
  .pword 0x7C6F
  .pword 0x7C78
  .pword 0x7C81
  .pword 0x7C8B
  .pword 0x7C94
  .pword 0x7C9D
  .pword 0x7CA5
  .pword 0x7CAE
  .pword 0x7CB7
  .pword 0x7CC0
  .pword 0x7CC8
  .pword 0x7CD1
  .pword 0x7CD9
  .pword 0x7CE2
  .pword 0x7CEA
  .pword 0x7CF3
  .pword 0x7CFB
  .pword 0x7D03
  .pword 0x7D0B
  .pword 0x7D13
  .pword 0x7D1B
  .pword 0x7D23
  .pword 0x7D2B
  .pword 0x7D33
  .pword 0x7D3B
  .pword 0x7D42
  .pword 0x7D4A
  .pword 0x7D51
  .pword 0x7D59
  .pword 0x7D60
  .pword 0x7D68
  .pword 0x7D6F
  .pword 0x7D76
  .pword 0x7D7E
  .pword 0x7D85
  .pword 0x7D8C
  .pword 0x7D93
  .pword 0x7D9A
  .pword 0x7DA1
  .pword 0x7DA8
  .pword 0x7DAE
  .pword 0x7DB5
  .pword 0x7DBC
  .pword 0x7DC2
  .pword 0x7DC9
  .pword 0x7DD0
  .pword 0x7DD6
  .pword 0x7DDC
  .pword 0x7DE3
  .pword 0x7DE9
  .pword 0x7DEF
  .pword 0x7DF5
  .pword 0x7DFC
  .pword 0x7E02
  .pword 0x7E08
  .pword 0x7E0E
  .pword 0x7E14
  .pword 0x7E1A
  .pword 0x7E1F
  .pword 0x7E25
  .pword 0x7E2B
  .pword 0x7E31
  .pword 0x7E36
  .pword 0x7E3C
  .pword 0x7E41
  .pword 0x7E47
  .pword 0x7E4C
  .pword 0x7E51
  .pword 0x7E57
  .pword 0x7E5C
  .pword 0x7E61
  .pword 0x7E66
  .pword 0x7E6C
  .pword 0x7E71
  .pword 0x7E76
  .pword 0x7E7B
  .pword 0x7E80
  .pword 0x7E85
  .pword 0x7E89
  .pword 0x7E8E
  .pword 0x7E93
  .pword 0x7E98
  .pword 0x7E9C
  .pword 0x7EA1
  .pword 0x7EA6
  .pword 0x7EAA
  .pword 0x7EAF
  .pword 0x7EB3
  .pword 0x7EB7
  .pword 0x7EBC
  .pword 0x7EC0
  .pword 0x7EC4
  .pword 0x7EC9
  .pword 0x7ECD
  .pword 0x7ED1
  .pword 0x7ED5
  .pword 0x7ED9
  .pword 0x7EDD
  .pword 0x7EE1
  .pword 0x7EE5
  .pword 0x7EE9
  .pword 0x7EED
  .pword 0x7EF1
  .pword 0x7EF4
  .pword 0x7EF8
  .pword 0x7EFC
  .pword 0x7F00
  .pword 0x7F03
  .pword 0x7F07
  .pword 0x7F0A
  .pword 0x7F0E
  .pword 0x7F11
  .pword 0x7F15
  .pword 0x7F18
  .pword 0x7F1C
  .pword 0x7F1F
  .pword 0x7F22
  .pword 0x7F25
  .pword 0x7F29
  .pword 0x7F2C
  .pword 0x7F2F
  .pword 0x7F32
  .pword 0x7F35
  .pword 0x7F38
  .pword 0x7F3B
  .pword 0x7F3E
  .pword 0x7F41
  .pword 0x7F44
  .pword 0x7F47
  .pword 0x7F4A
  .pword 0x7F4D
  .pword 0x7F4F
  .pword 0x7F52
  .pword 0x7F55
  .pword 0x7F58
  .pword 0x7F5A
  .pword 0x7F5D
  .pword 0x7F5F
  .pword 0x7F62
  .pword 0x7F64
  .pword 0x7F67
  .pword 0x7F69
  .pword 0x7F6C
  .pword 0x7F6E
  .pword 0x7F71
  .pword 0x7F73
  .pword 0x7F75
  .pword 0x7F78
  .pword 0x7F7A
  .pword 0x7F7C
  .pword 0x7F7E
  .pword 0x7F81
  .pword 0x7F83
  .pword 0x7F85
  .pword 0x7F87
  .pword 0x7F89
  .pword 0x7F8B
  .pword 0x7F8D
  .pword 0x7F8F
  .pword 0x7F91
  .pword 0x7F93
  .pword 0x7F95
  .pword 0x7F97
  .pword 0x7F99
  .pword 0x7F9A
  .pword 0x7F9C
  .pword 0x7F9E
  .pword 0x7FA0
  .pword 0x7FA2
  .pword 0x7FA3
  .pword 0x7FA5
  .pword 0x7FA7
  .pword 0x7FA8
  .pword 0x7FAA
  .pword 0x7FAC
  .pword 0x7FAD
  .pword 0x7FAF
  .pword 0x7FB0
  .pword 0x7FB2
  .pword 0x7FB3
  .pword 0x7FB5
  .pword 0x7FB6
  .pword 0x7FB8
  .pword 0x7FB9
  .pword 0x7FBA
  .pword 0x7FBC
  .pword 0x7FBD
  .pword 0x7FBE
  .pword 0x7FC0
  .pword 0x7FC1
  .pword 0x7FC2
  .pword 0x7FC3
  .pword 0x7FC5
  .pword 0x7FC6
  .pword 0x7FC7
  .pword 0x7FC8
  .pword 0x7FC9
  .pword 0x7FCB
  .pword 0x7FCC
  .pword 0x7FCD
  .pword 0x7FCE
  .pword 0x7FCF
  .pword 0x7FD0
  .pword 0x7FD1
  .pword 0x7FD2
  .pword 0x7FD3
  .pword 0x7FD4
  .pword 0x7FD5
  .pword 0x7FD6
  .pword 0x7FD7
  .pword 0x7FD8
  .pword 0x7FD9
  .pword 0x7FDA
  .pword 0x7FDA
  .pword 0x7FDB
  .pword 0x7FDC
  .pword 0x7FDD
  .pword 0x7FDE
  .pword 0x7FDF
  .pword 0x7FDF
  .pword 0x7FE0
  .pword 0x7FE1
  .pword 0x7FE2
  .pword 0x7FE2
  .pword 0x7FE3
  .pword 0x7FE4
  .pword 0x7FE4
  .pword 0x7FE5
  .pword 0x7FE6
  .pword 0x7FE6
  .pword 0x7FE7
  .pword 0x7FE8
  .pword 0x7FE8
  .pword 0x7FE9
  .pword 0x7FE9
  .pword 0x7FEA
  .pword 0x7FEB
  .pword 0x7FEB
  .pword 0x7FEC
  .pword 0x7FEC
  .pword 0x7FED
  .pword 0x7FED
  .pword 0x7FEE
  .pword 0x7FEE
  .pword 0x7FEF
  .pword 0x7FEF
  .pword 0x7FF0
  .pword 0x7FF0
  .pword 0x7FF0
  .pword 0x7FF1
  .pword 0x7FF1
  .pword 0x7FF2
  .pword 0x7FF2
  .pword 0x7FF3
  .pword 0x7FF3
  .pword 0x7FF3
  .pword 0x7FF4
  .pword 0x7FF4
  .pword 0x7FF4
  .pword 0x7FF5
  .pword 0x7FF5
  .pword 0x7FF5
  .pword 0x7FF6
  .pword 0x7FF6
  .pword 0x7FF6
  .pword 0x7FF7
  .pword 0x7FF7
  .pword 0x7FF7
  .pword 0x7FF7
  .pword 0x7FF8
  .pword 0x7FF8
  .pword 0x7FF8
  .pword 0x7FF8
  .pword 0x7FF9
  .pword 0x7FF9
  .pword 0x7FF9
  .pword 0x7FF9
  .pword 0x7FFA
  .pword 0x7FFA
  .pword 0x7FFA
  .pword 0x7FFA
  .pword 0x7FFA
  .pword 0x7FFA
  .pword 0x7FFB
  .pword 0x7FFB
  .pword 0x7FFB
  .pword 0x7FFB
  .pword 0x7FFB
  .pword 0x7FFB
  .pword 0x7FFC
  .pword 0x7FFC
  .pword 0x7FFC
  .pword 0x7FFC
  .pword 0x7FFC
  .pword 0x7FFC
  .pword 0x7FFC
  .pword 0x7FFD
  .pword 0x7FFD
  .pword 0x7FFD
  .pword 0x7FFD
  .pword 0x7FFD
  .pword 0x7FFD
  .pword 0x7FFD
  .pword 0x7FFD
  .pword 0x7FFD
  .pword 0x7FFD
  .pword 0x7FFD
  .pword 0x7FFE
  .pword 0x7FFE
  .pword 0x7FFE
  .pword 0x7FFE
  .pword 0x7FFE
  .pword 0x7FFE
  .pword 0x7FFE
  .pword 0x7FFE
  .pword 0x7FFE
  .pword 0x7FFE
  .pword 0x7FFE
  .pword 0x7FFE
  .pword 0x7FFE
  .pword 0x7FFE
  .pword 0x7FFE
  .pword 0x7FFE
  .pword 0x7FFE
  .pword 0x7FFE
  .pword 0x7FFF
  .pword 0x7FFF
  .pword 0x7FFF
  .pword 0x7FFF
  .pword 0x7FFF
  .pword 0x7FFF
  .pword 0x7FFF
  .pword 0x7FFF
  .pword 0x7FFF
  .pword 0x7FFF
  .pword 0x7FFF
  .pword 0x7FFF
  .pword 0x7FFF
  .pword 0x7FFF
  .pword 0x7FFF
  .pword 0x7FFF
  .pword 0x7FFF
  .pword 0x7FFF
  .pword 0x7FFF
  .pword 0x7FFF
  .pword 0x7FFF
  .pword 0x7FFF
  .pword 0x7FFF
  .pword 0x7FFF
  .pword 0x7FFF
  .pword 0x7FFF
  .pword 0x7FFF
  .pword 0x7FFF
  .pword 0x7FFF
  .pword 0x7FFF
  .pword 0x7FFF
  .pword 0x7FFF
  .pword 0x7FFF
  .pword 0x7FFF
  .pword 0x7FFF
  .pword 0x7FFF
  .pword 0x7FFF
  .pword 0x7FFF
  .pword 0x7FFF
  .pword 0x7FFF
  .pword 0x7FFF
  .pword 0x7FFF
  .pword 0x7FFF
  .pword 0x7FFF
  .pword 0x7FFF
  .pword 0x7FFF
  .pword 0x7FFF
  .pword 0x7FFF
  .pword 0x7FFF
  .pword 0x7FFF
  .pword 0x7FFF
  .pword 0x7FFF
  .pword 0x7FFF
  .pword 0x7FFF
  .pword 0x7FFF
  .pword 0x7FFF
  .pword 0x7FFF
  .pword 0x7FFF
  .pword 0x7FFF
  .pword 0x7FFF
  .pword 0x7FFF
  .pword 0x7FFF
  .pword 0x7FFF
  .pword 0x7FFF
  .pword 0x7FFF
  .pword 0x7FFF
  .pword 0x7FFF
  .pword 0x7FFF
  .pword 0x7FFF
  .pword 0x7FFF
  .pword 0x7FFF
  .pword 0x7FFF
  .pword 0x7FFF
  .pword 0x7FFF
  .pword 0x7FFF
  .pword 0x7FFF
  .pword 0x7FFF
  .pword 0x7FFF
  .pword 0x7FFF
  .pword 0x7FFF
  .pword 0x7FFF
  .pword 0x7FFF
  .pword 0x7FFF
  .pword 0x7FFF
  .pword 0x7FFF
  .pword 0x7FFF
  .pword 0x7FFF
  .pword 0x7FFF
  .pword 0x7FFF
  .pword 0x7FFF
  .pword 0x7FFF
  .pword 0x7FFF
  .pword 0x7FFF
  .pword 0x7FFF
  .pword 0x7FFF
  .pword 0x7FFF
  .pword 0x7FFF
  .pword 0x7FFF
  .pword 0x7FFF
  .pword 0x7FFF
  .pword 0x7FFF
  .pword 0x7FFF
  .pword 0x7FFF
  .pword 0x7FFF
  .pword 0x7FFF
  .pword 0x7FFF
  .pword 0x7FFF
  .pword 0x7FFF
  .pword 0x7FFF
  .pword 0x7FFF
  .pword 0x7FFF
  .pword 0x7FFF
  .pword 0x7FFF
  .pword 0x7FFF
  .pword 0x7FFF
  .pword 0x7FFE
  .pword 0x7FFE
  .pword 0x7FFE
  .pword 0x7FFE
  .pword 0x7FFE
  .pword 0x7FFE
  .pword 0x7FFE
  .pword 0x7FFE
  .pword 0x7FFE
  .pword 0x7FFE
  .pword 0x7FFE
  .pword 0x7FFE
  .pword 0x7FFE
  .pword 0x7FFE
  .pword 0x7FFE
  .pword 0x7FFE
  .pword 0x7FFE
  .pword 0x7FFE
  .pword 0x7FFD
  .pword 0x7FFD
  .pword 0x7FFD
  .pword 0x7FFD
  .pword 0x7FFD
  .pword 0x7FFD
  .pword 0x7FFD
  .pword 0x7FFD
  .pword 0x7FFD
  .pword 0x7FFD
  .pword 0x7FFD
  .pword 0x7FFC
  .pword 0x7FFC
  .pword 0x7FFC
  .pword 0x7FFC
  .pword 0x7FFC
  .pword 0x7FFC
  .pword 0x7FFC
  .pword 0x7FFB
  .pword 0x7FFB
  .pword 0x7FFB
  .pword 0x7FFB
  .pword 0x7FFB
  .pword 0x7FFB
  .pword 0x7FFA
  .pword 0x7FFA
  .pword 0x7FFA
  .pword 0x7FFA
  .pword 0x7FFA
  .pword 0x7FFA
  .pword 0x7FF9
  .pword 0x7FF9
  .pword 0x7FF9
  .pword 0x7FF9
  .pword 0x7FF8
  .pword 0x7FF8
  .pword 0x7FF8
  .pword 0x7FF8
  .pword 0x7FF7
  .pword 0x7FF7
  .pword 0x7FF7
  .pword 0x7FF7
  .pword 0x7FF6
  .pword 0x7FF6
  .pword 0x7FF6
  .pword 0x7FF5
  .pword 0x7FF5
  .pword 0x7FF5
  .pword 0x7FF4
  .pword 0x7FF4
  .pword 0x7FF4
  .pword 0x7FF3
  .pword 0x7FF3
  .pword 0x7FF3
  .pword 0x7FF2
  .pword 0x7FF2
  .pword 0x7FF1
  .pword 0x7FF1
  .pword 0x7FF0
  .pword 0x7FF0
  .pword 0x7FF0
  .pword 0x7FEF
  .pword 0x7FEF
  .pword 0x7FEE
  .pword 0x7FEE
  .pword 0x7FED
  .pword 0x7FED
  .pword 0x7FEC
  .pword 0x7FEC
  .pword 0x7FEB
  .pword 0x7FEB
  .pword 0x7FEA
  .pword 0x7FE9
  .pword 0x7FE9
  .pword 0x7FE8
  .pword 0x7FE8
  .pword 0x7FE7
  .pword 0x7FE6
  .pword 0x7FE6
  .pword 0x7FE5
  .pword 0x7FE4
  .pword 0x7FE4
  .pword 0x7FE3
  .pword 0x7FE2
  .pword 0x7FE2
  .pword 0x7FE1
  .pword 0x7FE0
  .pword 0x7FDF
  .pword 0x7FDF
  .pword 0x7FDE
  .pword 0x7FDD
  .pword 0x7FDC
  .pword 0x7FDB
  .pword 0x7FDA
  .pword 0x7FDA
  .pword 0x7FD9
  .pword 0x7FD8
  .pword 0x7FD7
  .pword 0x7FD6
  .pword 0x7FD5
  .pword 0x7FD4
  .pword 0x7FD3
  .pword 0x7FD2
  .pword 0x7FD1
  .pword 0x7FD0
  .pword 0x7FCF
  .pword 0x7FCE
  .pword 0x7FCD
  .pword 0x7FCC
  .pword 0x7FCB
  .pword 0x7FC9
  .pword 0x7FC8
  .pword 0x7FC7
  .pword 0x7FC6
  .pword 0x7FC5
  .pword 0x7FC3
  .pword 0x7FC2
  .pword 0x7FC1
  .pword 0x7FC0
  .pword 0x7FBE
  .pword 0x7FBD
  .pword 0x7FBC
  .pword 0x7FBA
  .pword 0x7FB9
  .pword 0x7FB8
  .pword 0x7FB6
  .pword 0x7FB5
  .pword 0x7FB3
  .pword 0x7FB2
  .pword 0x7FB0
  .pword 0x7FAF
  .pword 0x7FAD
  .pword 0x7FAC
  .pword 0x7FAA
  .pword 0x7FA8
  .pword 0x7FA7
  .pword 0x7FA5
  .pword 0x7FA3
  .pword 0x7FA2
  .pword 0x7FA0
  .pword 0x7F9E
  .pword 0x7F9C
  .pword 0x7F9A
  .pword 0x7F99
  .pword 0x7F97
  .pword 0x7F95
  .pword 0x7F93
  .pword 0x7F91
  .pword 0x7F8F
  .pword 0x7F8D
  .pword 0x7F8B
  .pword 0x7F89
  .pword 0x7F87
  .pword 0x7F85
  .pword 0x7F83
  .pword 0x7F81
  .pword 0x7F7E
  .pword 0x7F7C
  .pword 0x7F7A
  .pword 0x7F78
  .pword 0x7F75
  .pword 0x7F73
  .pword 0x7F71
  .pword 0x7F6E
  .pword 0x7F6C
  .pword 0x7F69
  .pword 0x7F67
  .pword 0x7F64
  .pword 0x7F62
  .pword 0x7F5F
  .pword 0x7F5D
  .pword 0x7F5A
  .pword 0x7F58
  .pword 0x7F55
  .pword 0x7F52
  .pword 0x7F4F
  .pword 0x7F4D
  .pword 0x7F4A
  .pword 0x7F47
  .pword 0x7F44
  .pword 0x7F41
  .pword 0x7F3E
  .pword 0x7F3B
  .pword 0x7F38
  .pword 0x7F35
  .pword 0x7F32
  .pword 0x7F2F
  .pword 0x7F2C
  .pword 0x7F29
  .pword 0x7F25
  .pword 0x7F22
  .pword 0x7F1F
  .pword 0x7F1C
  .pword 0x7F18
  .pword 0x7F15
  .pword 0x7F11
  .pword 0x7F0E
  .pword 0x7F0A
  .pword 0x7F07
  .pword 0x7F03
  .pword 0x7F00
  .pword 0x7EFC
  .pword 0x7EF8
  .pword 0x7EF4
  .pword 0x7EF1
  .pword 0x7EED
  .pword 0x7EE9
  .pword 0x7EE5
  .pword 0x7EE1
  .pword 0x7EDD
  .pword 0x7ED9
  .pword 0x7ED5
  .pword 0x7ED1
  .pword 0x7ECD
  .pword 0x7EC9
  .pword 0x7EC4
  .pword 0x7EC0
  .pword 0x7EBC
  .pword 0x7EB7
  .pword 0x7EB3
  .pword 0x7EAF
  .pword 0x7EAA
  .pword 0x7EA6
  .pword 0x7EA1
  .pword 0x7E9C
  .pword 0x7E98
  .pword 0x7E93
  .pword 0x7E8E
  .pword 0x7E89
  .pword 0x7E85
  .pword 0x7E80
  .pword 0x7E7B
  .pword 0x7E76
  .pword 0x7E71
  .pword 0x7E6C
  .pword 0x7E66
  .pword 0x7E61
  .pword 0x7E5C
  .pword 0x7E57
  .pword 0x7E51
  .pword 0x7E4C
  .pword 0x7E47
  .pword 0x7E41
  .pword 0x7E3C
  .pword 0x7E36
  .pword 0x7E31
  .pword 0x7E2B
  .pword 0x7E25
  .pword 0x7E1F
  .pword 0x7E1A
  .pword 0x7E14
  .pword 0x7E0E
  .pword 0x7E08
  .pword 0x7E02
  .pword 0x7DFC
  .pword 0x7DF5
  .pword 0x7DEF
  .pword 0x7DE9
  .pword 0x7DE3
  .pword 0x7DDC
  .pword 0x7DD6
  .pword 0x7DD0
  .pword 0x7DC9
  .pword 0x7DC2
  .pword 0x7DBC
  .pword 0x7DB5
  .pword 0x7DAE
  .pword 0x7DA8
  .pword 0x7DA1
  .pword 0x7D9A
  .pword 0x7D93
  .pword 0x7D8C
  .pword 0x7D85
  .pword 0x7D7E
  .pword 0x7D76
  .pword 0x7D6F
  .pword 0x7D68
  .pword 0x7D60
  .pword 0x7D59
  .pword 0x7D51
  .pword 0x7D4A
  .pword 0x7D42
  .pword 0x7D3B
  .pword 0x7D33
  .pword 0x7D2B
  .pword 0x7D23
  .pword 0x7D1B
  .pword 0x7D13
  .pword 0x7D0B
  .pword 0x7D03
  .pword 0x7CFB
  .pword 0x7CF3
  .pword 0x7CEA
  .pword 0x7CE2
  .pword 0x7CD9
  .pword 0x7CD1
  .pword 0x7CC8
  .pword 0x7CC0
  .pword 0x7CB7
  .pword 0x7CAE
  .pword 0x7CA5
  .pword 0x7C9D
  .pword 0x7C94
  .pword 0x7C8B
  .pword 0x7C81
  .pword 0x7C78
  .pword 0x7C6F
  .pword 0x7C66
  .pword 0x7C5C
  .pword 0x7C53
  .pword 0x7C49
  .pword 0x7C40
  .pword 0x7C36
  .pword 0x7C2C
  .pword 0x7C23
  .pword 0x7C19
  .pword 0x7C0F
  .pword 0x7C05
  .pword 0x7BFB
  .pword 0x7BF1
  .pword 0x7BE6
  .pword 0x7BDC
  .pword 0x7BD2
  .pword 0x7BC7
  .pword 0x7BBD
  .pword 0x7BB2
  .pword 0x7BA7
  .pword 0x7B9D
  .pword 0x7B92
  .pword 0x7B87
  .pword 0x7B7C
  .pword 0x7B71
  .pword 0x7B66
  .pword 0x7B5B
  .pword 0x7B4F
  .pword 0x7B44
  .pword 0x7B39
  .pword 0x7B2D
  .pword 0x7B21
  .pword 0x7B16
  .pword 0x7B0A
  .pword 0x7AFE
  .pword 0x7AF2
  .pword 0x7AE6
  .pword 0x7ADA
  .pword 0x7ACE
  .pword 0x7AC2
  .pword 0x7AB6
  .pword 0x7AA9
  .pword 0x7A9D
  .pword 0x7A90
  .pword 0x7A84
  .pword 0x7A77
  .pword 0x7A6A
  .pword 0x7A5D
  .pword 0x7A50
  .pword 0x7A43
  .pword 0x7A36
  .pword 0x7A29
  .pword 0x7A1B
  .pword 0x7A0E
  .pword 0x7A01
  .pword 0x79F3
  .pword 0x79E5
  .pword 0x79D8
  .pword 0x79CA
  .pword 0x79BC
  .pword 0x79AE
  .pword 0x79A0
  .pword 0x7992
  .pword 0x7983
  .pword 0x7975
  .pword 0x7967
  .pword 0x7958
  .pword 0x794A
  .pword 0x793B
  .pword 0x792C
  .pword 0x791D
  .pword 0x790E
  .pword 0x78FF
  .pword 0x78F0
  .pword 0x78E1
  .pword 0x78D1
  .pword 0x78C2
  .pword 0x78B2
  .pword 0x78A3
  .pword 0x7893
  .pword 0x7883
  .pword 0x7873
  .pword 0x7863
  .pword 0x7853
  .pword 0x7843
  .pword 0x7833
  .pword 0x7823
  .pword 0x7812
  .pword 0x7802
  .pword 0x77F1
  .pword 0x77E0
  .pword 0x77CF
  .pword 0x77BE
  .pword 0x77AD
  .pword 0x779C
  .pword 0x778B
  .pword 0x777A
  .pword 0x7768
  .pword 0x7757
  .pword 0x7745
  .pword 0x7733
  .pword 0x7721
  .pword 0x770F
  .pword 0x76FD
  .pword 0x76EB
  .pword 0x76D9
  .pword 0x76C7
  .pword 0x76B4
  .pword 0x76A2
  .pword 0x768F
  .pword 0x767C
  .pword 0x7669
  .pword 0x7656
  .pword 0x7643
  .pword 0x7630
  .pword 0x761D
  .pword 0x760A
  .pword 0x75F6
  .pword 0x75E3
  .pword 0x75CF
  .pword 0x75BB
  .pword 0x75A7
  .pword 0x7593
  .pword 0x757F
  .pword 0x756B
  .pword 0x7556
  .pword 0x7542
  .pword 0x752D
  .pword 0x7519
  .pword 0x7504
  .pword 0x74EF
  .pword 0x74DA
  .pword 0x74C5
  .pword 0x74B0
  .pword 0x749B
  .pword 0x7485
  .pword 0x7470
  .pword 0x745A
  .pword 0x7444
  .pword 0x742E
  .pword 0x7418
  .pword 0x7402
  .pword 0x73EC
  .pword 0x73D6
  .pword 0x73BF
  .pword 0x73A9
  .pword 0x7392
  .pword 0x737B
  .pword 0x7364
  .pword 0x734D
  .pword 0x7336
  .pword 0x731F
  .pword 0x7308
  .pword 0x72F0
  .pword 0x72D9
  .pword 0x72C1
  .pword 0x72A9
  .pword 0x7291
  .pword 0x7279
  .pword 0x7261
  .pword 0x7249
  .pword 0x7230
  .pword 0x7218
  .pword 0x71FF
  .pword 0x71E6
  .pword 0x71CD
  .pword 0x71B4
  .pword 0x719B
  .pword 0x7182
  .pword 0x7169
  .pword 0x714F
  .pword 0x7136
  .pword 0x711C
  .pword 0x7102
  .pword 0x70E8
  .pword 0x70CE
  .pword 0x70B4
  .pword 0x7099
  .pword 0x707F
  .pword 0x7064
  .pword 0x704A
  .pword 0x702F
  .pword 0x7014
  .pword 0x6FF9
  .pword 0x6FDE
  .pword 0x6FC2
  .pword 0x6FA7
  .pword 0x6F8B
  .pword 0x6F70
  .pword 0x6F54
  .pword 0x6F38
  .pword 0x6F1C
  .pword 0x6EFF
  .pword 0x6EE3
  .pword 0x6EC7
  .pword 0x6EAA
  .pword 0x6E8D
  .pword 0x6E70
  .pword 0x6E53
  .pword 0x6E36
  .pword 0x6E19
  .pword 0x6DFC
  .pword 0x6DDE
  .pword 0x6DC1
  .pword 0x6DA3
  .pword 0x6D85
  .pword 0x6D67
  .pword 0x6D49
  .pword 0x6D2A
  .pword 0x6D0C
  .pword 0x6CED
  .pword 0x6CCF
  .pword 0x6CB0
  .pword 0x6C91
  .pword 0x6C72
  .pword 0x6C53
  .pword 0x6C33
  .pword 0x6C14
  .pword 0x6BF4
  .pword 0x6BD4
  .pword 0x6BB4
  .pword 0x6B94
  .pword 0x6B74
  .pword 0x6B54
  .pword 0x6B34
  .pword 0x6B13
  .pword 0x6AF2
  .pword 0x6AD1
  .pword 0x6AB0
  .pword 0x6A8F
  .pword 0x6A6E
  .pword 0x6A4D
  .pword 0x6A2B
  .pword 0x6A09
  .pword 0x69E8
  .pword 0x69C6
  .pword 0x69A4
  .pword 0x6981
  .pword 0x695F
  .pword 0x693C
  .pword 0x691A
  .pword 0x68F7
  .pword 0x68D4
  .pword 0x68B1
  .pword 0x688E
  .pword 0x686A
  .pword 0x6847
  .pword 0x6823
  .pword 0x67FF
  .pword 0x67DC
  .pword 0x67B7
  .pword 0x6793
  .pword 0x676F
  .pword 0x674A
  .pword 0x6726
  .pword 0x6701
  .pword 0x66DC
  .pword 0x66B7
  .pword 0x6692
  .pword 0x666C
  .pword 0x6647
  .pword 0x6621
  .pword 0x65FB
  .pword 0x65D5
  .pword 0x65AF
  .pword 0x6589
  .pword 0x6563
  .pword 0x653C
  .pword 0x6515
  .pword 0x64EF
  .pword 0x64C8
  .pword 0x64A1
  .pword 0x6479
  .pword 0x6452
  .pword 0x642A
  .pword 0x6403
  .pword 0x63DB
  .pword 0x63B3
  .pword 0x638B
  .pword 0x6362
  .pword 0x633A
  .pword 0x6311
  .pword 0x62E8
  .pword 0x62BF
  .pword 0x6296
  .pword 0x626D
  .pword 0x6244
  .pword 0x621A
  .pword 0x61F1
  .pword 0x61C7
  .pword 0x619D
  .pword 0x6173
  .pword 0x6148
  .pword 0x611E
  .pword 0x60F3
  .pword 0x60C9
  .pword 0x609E
  .pword 0x6073
  .pword 0x6048
  .pword 0x601C
  .pword 0x5FF1
  .pword 0x5FC5
  .pword 0x5F99
  .pword 0x5F6D
  .pword 0x5F41
  .pword 0x5F15
  .pword 0x5EE8
  .pword 0x5EBC
  .pword 0x5E8F
  .pword 0x5E62
  .pword 0x5E35
  .pword 0x5E08
  .pword 0x5DDB
  .pword 0x5DAD
  .pword 0x5D7F
  .pword 0x5D52
  .pword 0x5D24
  .pword 0x5CF5
  .pword 0x5CC7
  .pword 0x5C99
  .pword 0x5C6A
  .pword 0x5C3B
  .pword 0x5C0C
  .pword 0x5BDD
  .pword 0x5BAE
  .pword 0x5B7E
  .pword 0x5B4F
  .pword 0x5B1F
  .pword 0x5AEF
  .pword 0x5ABF
  .pword 0x5A8F
  .pword 0x5A5F
  .pword 0x5A2E
  .pword 0x59FD
  .pword 0x59CD
  .pword 0x599B
  .pword 0x596A
  .pword 0x5939
  .pword 0x5907
  .pword 0x58D6
  .pword 0x58A4
  .pword 0x5872
  .pword 0x5840
  .pword 0x580E
  .pword 0x57DB
  .pword 0x57A8
  .pword 0x5776
  .pword 0x5743
  .pword 0x5710
  .pword 0x56DC
  .pword 0x56A9
  .pword 0x5675
  .pword 0x5641
  .pword 0x560D
  .pword 0x55D9
  .pword 0x55A5
  .pword 0x5571
  .pword 0x553C
  .pword 0x5507
  .pword 0x54D2
  .pword 0x549D
  .pword 0x5468
  .pword 0x5433
  .pword 0x53FD
  .pword 0x53C7
  .pword 0x5391
  .pword 0x535B
  .pword 0x5325
  .pword 0x52EF
  .pword 0x52B8
  .pword 0x5281
  .pword 0x524A
  .pword 0x5213
  .pword 0x51DC
  .pword 0x51A5
  .pword 0x516D
  .pword 0x5135
  .pword 0x50FD
  .pword 0x50C5
  .pword 0x508D
  .pword 0x5055
  .pword 0x501C
  .pword 0x4FE3
  .pword 0x4FAA
  .pword 0x4F71
  .pword 0x4F38
  .pword 0x4EFF
  .pword 0x4EC5
  .pword 0x4E8B
  .pword 0x4E51
  .pword 0x4E17
  .pword 0x4DDD
  .pword 0x4DA3
  .pword 0x4D68
  .pword 0x4D2D
  .pword 0x4CF2
  .pword 0x4CB7
  .pword 0x4C7C
  .pword 0x4C41
  .pword 0x4C05
  .pword 0x4BC9
  .pword 0x4B8D
  .pword 0x4B51
  .pword 0x4B15
  .pword 0x4AD8
  .pword 0x4A9C
  .pword 0x4A5F
  .pword 0x4A22
  .pword 0x49E5
  .pword 0x49A8
  .pword 0x496A
  .pword 0x492C
  .pword 0x48EF
  .pword 0x48B1
  .pword 0x4872
  .pword 0x4834
  .pword 0x47F6
  .pword 0x47B7
  .pword 0x4778
  .pword 0x4739
  .pword 0x46FA
  .pword 0x46BA
  .pword 0x467B
  .pword 0x463B
  .pword 0x45FB
  .pword 0x45BB
  .pword 0x457B
  .pword 0x453B
  .pword 0x44FA
  .pword 0x44B9
  .pword 0x4478
  .pword 0x4437
  .pword 0x43F6
  .pword 0x43B5
  .pword 0x4373
  .pword 0x4331
  .pword 0x42EF
  .pword 0x42AD
  .pword 0x426B
  .pword 0x4228
  .pword 0x41E6
  .pword 0x41A3
  .pword 0x4160
  .pword 0x411D
  .pword 0x40DA
  .pword 0x4096
  .pword 0x4052
  .pword 0x400F
  .pword 0x3FCB
  .pword 0x3F86
  .pword 0x3F42
  .pword 0x3EFE
  .pword 0x3EB9
  .pword 0x3E74
  .pword 0x3E2F
  .pword 0x3DEA
  .pword 0x3DA4
  .pword 0x3D5F
  .pword 0x3D19
  .pword 0x3CD3
  .pword 0x3C8D
  .pword 0x3C47
  .pword 0x3C00
  .pword 0x3BBA
  .pword 0x3B73
  .pword 0x3B2C
  .pword 0x3AE5
  .pword 0x3A9D
  .pword 0x3A56
  .pword 0x3A0E
  .pword 0x39C6
  .pword 0x397E
  .pword 0x3936
  .pword 0x38EE
  .pword 0x38A5
  .pword 0x385D
  .pword 0x3814
  .pword 0x37CB
  .pword 0x3782
  .pword 0x3738
  .pword 0x36EF
  .pword 0x36A5
  .pword 0x365B
  .pword 0x3611
  .pword 0x35C7
  .pword 0x357C
  .pword 0x3532
  .pword 0x34E7
  .pword 0x349C
  .pword 0x3451
  .pword 0x3406
  .pword 0x33BA
  .pword 0x336E
  .pword 0x3323
  .pword 0x32D7
  .pword 0x328A
  .pword 0x323E
  .pword 0x31F2
  .pword 0x31A5
  .pword 0x3158
  .pword 0x310B
  .pword 0x30BE
  .pword 0x3070
  .pword 0x3023
  .pword 0x2FD5
  .pword 0x2F87
  .pword 0x2F39
  .pword 0x2EEB
  .pword 0x2E9D
  .pword 0x2E4E
  .pword 0x2DFF
  .pword 0x2DB0
  .pword 0x2D61
  .pword 0x2D12
  .pword 0x2CC2
  .pword 0x2C73
  .pword 0x2C23
  .pword 0x2BD3
  .pword 0x2B83
  .pword 0x2B33
  .pword 0x2AE2
  .pword 0x2A92
  .pword 0x2A41
  .pword 0x29F0
  .pword 0x299F
  .pword 0x294D
  .pword 0x28FC
  .pword 0x28AA
  .pword 0x2858
  .pword 0x2806
  .pword 0x27B4
  .pword 0x2762
  .pword 0x270F
  .pword 0x26BC
  .pword 0x2669
  .pword 0x2616
  .pword 0x25C3
  .pword 0x2570
  .pword 0x251C
  .pword 0x24C8
  .pword 0x2474
  .pword 0x2420
  .pword 0x23CC
  .pword 0x2378
  .pword 0x2323
  .pword 0x22CE
  .pword 0x2279
  .pword 0x2224
  .pword 0x21CF
  .pword 0x217A
  .pword 0x2124
  .pword 0x20CE
  .pword 0x2078
  .pword 0x2022
  .pword 0x1FCC
  .pword 0x1F75
  .pword 0x1F1F
  .pword 0x1EC8
  .pword 0x1E71
  .pword 0x1E1A
  .pword 0x1DC3
  .pword 0x1D6B
  .pword 0x1D13
  .pword 0x1CBC
  .pword 0x1C64
  .pword 0x1C0B
  .pword 0x1BB3
  .pword 0x1B5B
  .pword 0x1B02
  .pword 0x1AA9
  .pword 0x1A50
  .pword 0x19F7
  .pword 0x199E
  .pword 0x1944
  .pword 0x18EB
  .pword 0x1891
  .pword 0x1837
  .pword 0x17DD
  .pword 0x1782
  .pword 0x1728
  .pword 0x16CD
  .pword 0x1672
  .pword 0x1617
  .pword 0x15BC
  .pword 0x1561
  .pword 0x1505
  .pword 0x14AA
  .pword 0x144E
  .pword 0x13F2
  .pword 0x1396
  .pword 0x133A
  .pword 0x12DD
  .pword 0x1280
  .pword 0x1224
  .pword 0x11C7
  .pword 0x116A
  .pword 0x110C
  .pword 0x10AF
  .pword 0x1051
  .pword 0x0FF3
  .pword 0x0F95
  .pword 0x0F37
  .pword 0x0ED9
  .pword 0x0E7B
  .pword 0x0E1C
  .pword 0x0DBD
  .pword 0x0D5E
  .pword 0x0CFF
  .pword 0x0CA0
  .pword 0x0C41
  .pword 0x0BE1
  .pword 0x0B81
  .pword 0x0B22
  .pword 0x0AC2
  .pword 0x0A61
  .pword 0x0A01
  .pword 0x09A0
  .pword 0x0940
  .pword 0x08DF
  .pword 0x087E
  .pword 0x081D
  .pword 0x07BB
  .pword 0x075A
  .pword 0x06F8
  .pword 0x0697
  .pword 0x0635
  .pword 0x05D2
  .pword 0x0570
  .pword 0x050E
  .pword 0x04AB
  .pword 0x0448
  .pword 0x03E6
  .pword 0x0382
  .pword 0x031F
  .pword 0x02BC
  .pword 0x0258
  .pword 0x01F5
  .pword 0x0191
  .pword 0x012D
  .pword 0x00C9
  .pword 0x0064
  .pword 0x0000
  .pword 0xFF9D
  .pword 0xFF38
  .pword 0xFED4
  .pword 0xFE70
  .pword 0xFE0C
  .pword 0xFDA9
  .pword 0xFD45
  .pword 0xFCE2
  .pword 0xFC7F
  .pword 0xFC1B
  .pword 0xFBB9
  .pword 0xFB56
  .pword 0xFAF3
  .pword 0xFA91
  .pword 0xFA2F
  .pword 0xF9CC
  .pword 0xF96A
  .pword 0xF909
  .pword 0xF8A7
  .pword 0xF846
  .pword 0xF7E4
  .pword 0xF783
  .pword 0xF722
  .pword 0xF6C1
  .pword 0xF661
  .pword 0xF600
  .pword 0xF5A0
  .pword 0xF53F
  .pword 0xF4DF
  .pword 0xF480
  .pword 0xF420
  .pword 0xF3C0
  .pword 0xF361
  .pword 0xF302
  .pword 0xF2A3
  .pword 0xF244
  .pword 0xF1E5
  .pword 0xF186
  .pword 0xF128
  .pword 0xF0CA
  .pword 0xF06C
  .pword 0xF00E
  .pword 0xEFB0
  .pword 0xEF52
  .pword 0xEEF5
  .pword 0xEE97
  .pword 0xEE3A
  .pword 0xEDDD
  .pword 0xED81
  .pword 0xED24
  .pword 0xECC7
  .pword 0xEC6B
  .pword 0xEC0F
  .pword 0xEBB3
  .pword 0xEB57
  .pword 0xEAFC
  .pword 0xEAA0
  .pword 0xEA45
  .pword 0xE9EA
  .pword 0xE98F
  .pword 0xE934
  .pword 0xE8D9
  .pword 0xE87F
  .pword 0xE824
  .pword 0xE7CA
  .pword 0xE770
  .pword 0xE716
  .pword 0xE6BD
  .pword 0xE663
  .pword 0xE60A
  .pword 0xE5B1
  .pword 0xE558
  .pword 0xE4FF
  .pword 0xE4A6
  .pword 0xE44E
  .pword 0xE3F6
  .pword 0xE39D
  .pword 0xE345
  .pword 0xE2EE
  .pword 0xE296
  .pword 0xE23E
  .pword 0xE1E7
  .pword 0xE190
  .pword 0xE139
  .pword 0xE0E2
  .pword 0xE08C
  .pword 0xE035
  .pword 0xDFDF
  .pword 0xDF89
  .pword 0xDF33
  .pword 0xDEDD
  .pword 0xDE87
  .pword 0xDE32
  .pword 0xDDDD
  .pword 0xDD88
  .pword 0xDD33
  .pword 0xDCDE
  .pword 0xDC89
  .pword 0xDC35
  .pword 0xDBE1
  .pword 0xDB8D
  .pword 0xDB39
  .pword 0xDAE5
  .pword 0xDA91
  .pword 0xDA3E
  .pword 0xD9EB
  .pword 0xD998
  .pword 0xD945
  .pword 0xD8F2
  .pword 0xD89F
  .pword 0xD84D
  .pword 0xD7FB
  .pword 0xD7A9
  .pword 0xD757
  .pword 0xD705
  .pword 0xD6B4
  .pword 0xD662
  .pword 0xD611
  .pword 0xD5C0
  .pword 0xD56F
  .pword 0xD51F
  .pword 0xD4CE
  .pword 0xD47E
  .pword 0xD42E
  .pword 0xD3DE
  .pword 0xD38E
  .pword 0xD33F
  .pword 0xD2EF
  .pword 0xD2A0
  .pword 0xD251
  .pword 0xD202
  .pword 0xD1B3
  .pword 0xD164
  .pword 0xD116
  .pword 0xD0C8
  .pword 0xD07A
  .pword 0xD02C
  .pword 0xCFDE
  .pword 0xCF91
  .pword 0xCF43
  .pword 0xCEF6
  .pword 0xCEA9
  .pword 0xCE5C
  .pword 0xCE0F
  .pword 0xCDC3
  .pword 0xCD77
  .pword 0xCD2A
  .pword 0xCCDE
  .pword 0xCC93
  .pword 0xCC47
  .pword 0xCBFB
  .pword 0xCBB0
  .pword 0xCB65
  .pword 0xCB1A
  .pword 0xCACF
  .pword 0xCA85
  .pword 0xCA3A
  .pword 0xC9F0
  .pword 0xC9A6
  .pword 0xC95C
  .pword 0xC912
  .pword 0xC8C9
  .pword 0xC87F
  .pword 0xC836
  .pword 0xC7ED
  .pword 0xC7A4
  .pword 0xC75C
  .pword 0xC713
  .pword 0xC6CB
  .pword 0xC683
  .pword 0xC63B
  .pword 0xC5F3
  .pword 0xC5AB
  .pword 0xC564
  .pword 0xC51C
  .pword 0xC4D5
  .pword 0xC48E
  .pword 0xC447
  .pword 0xC401
  .pword 0xC3BA
  .pword 0xC374
  .pword 0xC32E
  .pword 0xC2E8
  .pword 0xC2A2
  .pword 0xC25D
  .pword 0xC217
  .pword 0xC1D2
  .pword 0xC18D
  .pword 0xC148
  .pword 0xC103
  .pword 0xC0BF
  .pword 0xC07B
  .pword 0xC036
  .pword 0xBFF2
  .pword 0xBFAF
  .pword 0xBF6B
  .pword 0xBF27
  .pword 0xBEE4
  .pword 0xBEA1
  .pword 0xBE5E
  .pword 0xBE1B
  .pword 0xBDD9
  .pword 0xBD96
  .pword 0xBD54
  .pword 0xBD12
  .pword 0xBCD0
  .pword 0xBC8E
  .pword 0xBC4C
  .pword 0xBC0B
  .pword 0xBBCA
  .pword 0xBB89
  .pword 0xBB48
  .pword 0xBB07
  .pword 0xBAC6
  .pword 0xBA86
  .pword 0xBA46
  .pword 0xBA06
  .pword 0xB9C6
  .pword 0xB986
  .pword 0xB947
  .pword 0xB907
  .pword 0xB8C8
  .pword 0xB889
  .pword 0xB84A
  .pword 0xB80B
  .pword 0xB7CD
  .pword 0xB78F
  .pword 0xB750
  .pword 0xB712
  .pword 0xB6D5
  .pword 0xB697
  .pword 0xB659
  .pword 0xB61C
  .pword 0xB5DF
  .pword 0xB5A2
  .pword 0xB565
  .pword 0xB529
  .pword 0xB4EC
  .pword 0xB4B0
  .pword 0xB474
  .pword 0xB438
  .pword 0xB3FC
  .pword 0xB3C0
  .pword 0xB385
  .pword 0xB34A
  .pword 0xB30F
  .pword 0xB2D4
  .pword 0xB299
  .pword 0xB25E
  .pword 0xB224
  .pword 0xB1EA
  .pword 0xB1B0
  .pword 0xB176
  .pword 0xB13C
  .pword 0xB102
  .pword 0xB0C9
  .pword 0xB090
  .pword 0xB057
  .pword 0xB01E
  .pword 0xAFE5
  .pword 0xAFAC
  .pword 0xAF74
  .pword 0xAF3C
  .pword 0xAF04
  .pword 0xAECC
  .pword 0xAE94
  .pword 0xAE5C
  .pword 0xAE25
  .pword 0xADEE
  .pword 0xADB7
  .pword 0xAD80
  .pword 0xAD49
  .pword 0xAD12
  .pword 0xACDC
  .pword 0xACA6
  .pword 0xAC70
  .pword 0xAC3A
  .pword 0xAC04
  .pword 0xABCE
  .pword 0xAB99
  .pword 0xAB64
  .pword 0xAB2F
  .pword 0xAAFA
  .pword 0xAAC5
  .pword 0xAA90
  .pword 0xAA5C
  .pword 0xAA28
  .pword 0xA9F4
  .pword 0xA9C0
  .pword 0xA98C
  .pword 0xA958
  .pword 0xA925
  .pword 0xA8F1
  .pword 0xA8BE
  .pword 0xA88B
  .pword 0xA859
  .pword 0xA826
  .pword 0xA7F3
  .pword 0xA7C1
  .pword 0xA78F
  .pword 0xA75D
  .pword 0xA72B
  .pword 0xA6FA
  .pword 0xA6C8
  .pword 0xA697
  .pword 0xA666
  .pword 0xA634
  .pword 0xA604
  .pword 0xA5D3
  .pword 0xA5A2
  .pword 0xA572
  .pword 0xA542
  .pword 0xA512
  .pword 0xA4E2
  .pword 0xA4B2
  .pword 0xA483
  .pword 0xA453
  .pword 0xA424
  .pword 0xA3F5
  .pword 0xA3C6
  .pword 0xA397
  .pword 0xA368
  .pword 0xA33A
  .pword 0xA30C
  .pword 0xA2DD
  .pword 0xA2AF
  .pword 0xA282
  .pword 0xA254
  .pword 0xA226
  .pword 0xA1F9
  .pword 0xA1CC
  .pword 0xA19F
  .pword 0xA172
  .pword 0xA145
  .pword 0xA119
  .pword 0xA0EC
  .pword 0xA0C0
  .pword 0xA094
  .pword 0xA068
  .pword 0xA03C
  .pword 0xA010
  .pword 0x9FE5
  .pword 0x9FB9
  .pword 0x9F8E
  .pword 0x9F63
  .pword 0x9F38
  .pword 0x9F0E
  .pword 0x9EE3
  .pword 0x9EB9
  .pword 0x9E8E
  .pword 0x9E64
  .pword 0x9E3A
  .pword 0x9E10
  .pword 0x9DE7
  .pword 0x9DBD
  .pword 0x9D94
  .pword 0x9D6B
  .pword 0x9D42
  .pword 0x9D19
  .pword 0x9CF0
  .pword 0x9CC7
  .pword 0x9C9F
  .pword 0x9C76
  .pword 0x9C4E
  .pword 0x9C26
  .pword 0x9BFE
  .pword 0x9BD7
  .pword 0x9BAF
  .pword 0x9B88
  .pword 0x9B60
  .pword 0x9B39
  .pword 0x9B12
  .pword 0x9AEC
  .pword 0x9AC5
  .pword 0x9A9E
  .pword 0x9A78
  .pword 0x9A52
  .pword 0x9A2C
  .pword 0x9A06
  .pword 0x99E0
  .pword 0x99BA
  .pword 0x9995
  .pword 0x996F
  .pword 0x994A
  .pword 0x9925
  .pword 0x9900
  .pword 0x98DB
  .pword 0x98B7
  .pword 0x9892
  .pword 0x986E
  .pword 0x984A
  .pword 0x9825
  .pword 0x9802
  .pword 0x97DE
  .pword 0x97BA
  .pword 0x9797
  .pword 0x9773
  .pword 0x9750
  .pword 0x972D
  .pword 0x970A
  .pword 0x96E7
  .pword 0x96C5
  .pword 0x96A2
  .pword 0x9680
  .pword 0x965D
  .pword 0x963B
  .pword 0x9619
  .pword 0x95F8
  .pword 0x95D6
  .pword 0x95B4
  .pword 0x9593
  .pword 0x9572
  .pword 0x9551
  .pword 0x9530
  .pword 0x950F
  .pword 0x94EE
  .pword 0x94CD
  .pword 0x94AD
  .pword 0x948D
  .pword 0x946D
  .pword 0x944D
  .pword 0x942D
  .pword 0x940D
  .pword 0x93ED
  .pword 0x93CE
  .pword 0x93AE
  .pword 0x938F
  .pword 0x9370
  .pword 0x9351
  .pword 0x9332
  .pword 0x9314
  .pword 0x92F5
  .pword 0x92D7
  .pword 0x92B8
  .pword 0x929A
  .pword 0x927C
  .pword 0x925E
  .pword 0x9240
  .pword 0x9223
  .pword 0x9205
  .pword 0x91E8
  .pword 0x91CB
  .pword 0x91AE
  .pword 0x9191
  .pword 0x9174
  .pword 0x9157
  .pword 0x913A
  .pword 0x911E
  .pword 0x9102
  .pword 0x90E5
  .pword 0x90C9
  .pword 0x90AD
  .pword 0x9091
  .pword 0x9076
  .pword 0x905A
  .pword 0x903F
  .pword 0x9023
  .pword 0x9008
  .pword 0x8FED
  .pword 0x8FD2
  .pword 0x8FB7
  .pword 0x8F9D
  .pword 0x8F82
  .pword 0x8F68
  .pword 0x8F4D
  .pword 0x8F33
  .pword 0x8F19
  .pword 0x8EFF
  .pword 0x8EE5
  .pword 0x8ECB
  .pword 0x8EB2
  .pword 0x8E98
  .pword 0x8E7F
  .pword 0x8E66
  .pword 0x8E4D
  .pword 0x8E34
  .pword 0x8E1B
  .pword 0x8E02
  .pword 0x8DE9
  .pword 0x8DD1
  .pword 0x8DB8
  .pword 0x8DA0
  .pword 0x8D88
  .pword 0x8D70
  .pword 0x8D58
  .pword 0x8D40
  .pword 0x8D28
  .pword 0x8D11
  .pword 0x8CF9
  .pword 0x8CE2
  .pword 0x8CCB
  .pword 0x8CB4
  .pword 0x8C9D
  .pword 0x8C86
  .pword 0x8C6F
  .pword 0x8C58
  .pword 0x8C42
  .pword 0x8C2B
  .pword 0x8C15
  .pword 0x8BFF
  .pword 0x8BE9
  .pword 0x8BD3
  .pword 0x8BBD
  .pword 0x8BA7
  .pword 0x8B91
  .pword 0x8B7C
  .pword 0x8B66
  .pword 0x8B51
  .pword 0x8B3C
  .pword 0x8B27
  .pword 0x8B12
  .pword 0x8AFD
  .pword 0x8AE8
  .pword 0x8AD4
  .pword 0x8ABF
  .pword 0x8AAB
  .pword 0x8A96
  .pword 0x8A82
  .pword 0x8A6E
  .pword 0x8A5A
  .pword 0x8A46
  .pword 0x8A32
  .pword 0x8A1E
  .pword 0x8A0B
  .pword 0x89F7
  .pword 0x89E4
  .pword 0x89D1
  .pword 0x89BE
  .pword 0x89AB
  .pword 0x8998
  .pword 0x8985
  .pword 0x8972
  .pword 0x895F
  .pword 0x894D
  .pword 0x893A
  .pword 0x8928
  .pword 0x8916
  .pword 0x8904
  .pword 0x88F2
  .pword 0x88E0
  .pword 0x88CE
  .pword 0x88BC
  .pword 0x88AA
  .pword 0x8899
  .pword 0x8887
  .pword 0x8876
  .pword 0x8865
  .pword 0x8854
  .pword 0x8843
  .pword 0x8832
  .pword 0x8821
  .pword 0x8810
  .pword 0x87FF
  .pword 0x87EF
  .pword 0x87DE
  .pword 0x87CE
  .pword 0x87BE
  .pword 0x87AE
  .pword 0x879E
  .pword 0x878E
  .pword 0x877E
  .pword 0x876E
  .pword 0x875E
  .pword 0x874F
  .pword 0x873F
  .pword 0x8730
  .pword 0x8720
  .pword 0x8711
  .pword 0x8702
  .pword 0x86F3
  .pword 0x86E4
  .pword 0x86D5
  .pword 0x86C6
  .pword 0x86B7
  .pword 0x86A9
  .pword 0x869A
  .pword 0x868C
  .pword 0x867E
  .pword 0x866F
  .pword 0x8661
  .pword 0x8653
  .pword 0x8645
  .pword 0x8637
  .pword 0x8629
  .pword 0x861C
  .pword 0x860E
  .pword 0x8600
  .pword 0x85F3
  .pword 0x85E6
  .pword 0x85D8
  .pword 0x85CB
  .pword 0x85BE
  .pword 0x85B1
  .pword 0x85A4
  .pword 0x8597
  .pword 0x858A
  .pword 0x857D
  .pword 0x8571
  .pword 0x8564
  .pword 0x8558
  .pword 0x854B
  .pword 0x853F
  .pword 0x8533
  .pword 0x8527
  .pword 0x851B
  .pword 0x850F
  .pword 0x8503
  .pword 0x84F7
  .pword 0x84EB
  .pword 0x84E0
  .pword 0x84D4
  .pword 0x84C8
  .pword 0x84BD
  .pword 0x84B2
  .pword 0x84A6
  .pword 0x849B
  .pword 0x8490
  .pword 0x8485
  .pword 0x847A
  .pword 0x846F
  .pword 0x8464
  .pword 0x845A
  .pword 0x844F
  .pword 0x8444
  .pword 0x843A
  .pword 0x842F
  .pword 0x8425
  .pword 0x841B
  .pword 0x8410
  .pword 0x8406
  .pword 0x83FC
  .pword 0x83F2
  .pword 0x83E8
  .pword 0x83DE
  .pword 0x83D5
  .pword 0x83CB
  .pword 0x83C1
  .pword 0x83B8
  .pword 0x83AE
  .pword 0x83A5
  .pword 0x839B
  .pword 0x8392
  .pword 0x8389
  .pword 0x8380
  .pword 0x8376
  .pword 0x836D
  .pword 0x8364
  .pword 0x835C
  .pword 0x8353
  .pword 0x834A
  .pword 0x8341
  .pword 0x8339
  .pword 0x8330
  .pword 0x8328
  .pword 0x831F
  .pword 0x8317
  .pword 0x830E
  .pword 0x8306
  .pword 0x82FE
  .pword 0x82F6
  .pword 0x82EE
  .pword 0x82E6
  .pword 0x82DE
  .pword 0x82D6
  .pword 0x82CE
  .pword 0x82C6
  .pword 0x82BF
  .pword 0x82B7
  .pword 0x82B0
  .pword 0x82A8
  .pword 0x82A1
  .pword 0x8299
  .pword 0x8292
  .pword 0x828B
  .pword 0x8283
  .pword 0x827C
  .pword 0x8275
  .pword 0x826E
  .pword 0x8267
  .pword 0x8260
  .pword 0x8259
  .pword 0x8253
  .pword 0x824C
  .pword 0x8245
  .pword 0x823F
  .pword 0x8238
  .pword 0x8231
  .pword 0x822B
  .pword 0x8225
  .pword 0x821E
  .pword 0x8218
  .pword 0x8212
  .pword 0x820C
  .pword 0x8205
  .pword 0x81FF
  .pword 0x81F9
  .pword 0x81F3
  .pword 0x81ED
  .pword 0x81E7
  .pword 0x81E2
  .pword 0x81DC
  .pword 0x81D6
  .pword 0x81D0
  .pword 0x81CB
  .pword 0x81C5
  .pword 0x81C0
  .pword 0x81BA
  .pword 0x81B5
  .pword 0x81B0
  .pword 0x81AA
  .pword 0x81A5
  .pword 0x81A0
  .pword 0x819B
  .pword 0x8195
  .pword 0x8190
  .pword 0x818B
  .pword 0x8186
  .pword 0x8181
  .pword 0x817C
  .pword 0x8178
  .pword 0x8173
  .pword 0x816E
  .pword 0x8169
  .pword 0x8165
  .pword 0x8160
  .pword 0x815B
  .pword 0x8157
  .pword 0x8152
  .pword 0x814E
  .pword 0x814A
  .pword 0x8145
  .pword 0x8141
  .pword 0x813D
  .pword 0x8138
  .pword 0x8134
  .pword 0x8130
  .pword 0x812C
  .pword 0x8128
  .pword 0x8124
  .pword 0x8120
  .pword 0x811C
  .pword 0x8118
  .pword 0x8114
  .pword 0x8110
  .pword 0x810D
  .pword 0x8109
  .pword 0x8105
  .pword 0x8101
  .pword 0x80FE
  .pword 0x80FA
  .pword 0x80F7
  .pword 0x80F3
  .pword 0x80F0
  .pword 0x80EC
  .pword 0x80E9
  .pword 0x80E5
  .pword 0x80E2
  .pword 0x80DF
  .pword 0x80DC
  .pword 0x80D8
  .pword 0x80D5
  .pword 0x80D2
  .pword 0x80CF
  .pword 0x80CC
  .pword 0x80C9
  .pword 0x80C6
  .pword 0x80C3
  .pword 0x80C0
  .pword 0x80BD
  .pword 0x80BA
  .pword 0x80B7
  .pword 0x80B4
  .pword 0x80B2
  .pword 0x80AF
  .pword 0x80AC
  .pword 0x80A9
  .pword 0x80A7
  .pword 0x80A4
  .pword 0x80A2
  .pword 0x809F
  .pword 0x809D
  .pword 0x809A
  .pword 0x8098
  .pword 0x8095
  .pword 0x8093
  .pword 0x8090
  .pword 0x808E
  .pword 0x808C
  .pword 0x8089
  .pword 0x8087
  .pword 0x8085
  .pword 0x8083
  .pword 0x8080
  .pword 0x807E
  .pword 0x807C
  .pword 0x807A
  .pword 0x8078
  .pword 0x8076
  .pword 0x8074
  .pword 0x8072
  .pword 0x8070
  .pword 0x806E
  .pword 0x806C
  .pword 0x806A
  .pword 0x8068
  .pword 0x8067
  .pword 0x8065
  .pword 0x8063
  .pword 0x8061
  .pword 0x805F
  .pword 0x805E
  .pword 0x805C
  .pword 0x805A
  .pword 0x8059
  .pword 0x8057
  .pword 0x8055
  .pword 0x8054
  .pword 0x8052
  .pword 0x8051
  .pword 0x804F
  .pword 0x804E
  .pword 0x804C
  .pword 0x804B
  .pword 0x8049
  .pword 0x8048
  .pword 0x8047
  .pword 0x8045
  .pword 0x8044
  .pword 0x8043
  .pword 0x8041
  .pword 0x8040
  .pword 0x803F
  .pword 0x803E
  .pword 0x803C
  .pword 0x803B
  .pword 0x803A
  .pword 0x8039
  .pword 0x8038
  .pword 0x8036
  .pword 0x8035
  .pword 0x8034
  .pword 0x8033
  .pword 0x8032
  .pword 0x8031
  .pword 0x8030
  .pword 0x802F
  .pword 0x802E
  .pword 0x802D
  .pword 0x802C
  .pword 0x802B
  .pword 0x802A
  .pword 0x8029
  .pword 0x8028
  .pword 0x8027
  .pword 0x8027
  .pword 0x8026
  .pword 0x8025
  .pword 0x8024
  .pword 0x8023
  .pword 0x8022
  .pword 0x8022
  .pword 0x8021
  .pword 0x8020
  .pword 0x801F
  .pword 0x801F
  .pword 0x801E
  .pword 0x801D
  .pword 0x801D
  .pword 0x801C
  .pword 0x801B
  .pword 0x801B
  .pword 0x801A
  .pword 0x8019
  .pword 0x8019
  .pword 0x8018
  .pword 0x8018
  .pword 0x8017
  .pword 0x8016
  .pword 0x8016
  .pword 0x8015
  .pword 0x8015
  .pword 0x8014
  .pword 0x8014
  .pword 0x8013
  .pword 0x8013
  .pword 0x8012
  .pword 0x8012
  .pword 0x8011
  .pword 0x8011
  .pword 0x8011
  .pword 0x8010
  .pword 0x8010
  .pword 0x800F
  .pword 0x800F
  .pword 0x800E
  .pword 0x800E
  .pword 0x800E
  .pword 0x800D
  .pword 0x800D
  .pword 0x800D
  .pword 0x800C
  .pword 0x800C
  .pword 0x800C
  .pword 0x800B
  .pword 0x800B
  .pword 0x800B
  .pword 0x800A
  .pword 0x800A
  .pword 0x800A
  .pword 0x800A
  .pword 0x8009
  .pword 0x8009
  .pword 0x8009
  .pword 0x8009
  .pword 0x8008
  .pword 0x8008
  .pword 0x8008
  .pword 0x8008
  .pword 0x8007
  .pword 0x8007
  .pword 0x8007
  .pword 0x8007
  .pword 0x8007
  .pword 0x8007
  .pword 0x8006
  .pword 0x8006
  .pword 0x8006
  .pword 0x8006
  .pword 0x8006
  .pword 0x8006
  .pword 0x8005
  .pword 0x8005
  .pword 0x8005
  .pword 0x8005
  .pword 0x8005
  .pword 0x8005
  .pword 0x8005
  .pword 0x8004
  .pword 0x8004
  .pword 0x8004
  .pword 0x8004
  .pword 0x8004
  .pword 0x8004
  .pword 0x8004
  .pword 0x8004
  .pword 0x8004
  .pword 0x8004
  .pword 0x8004
  .pword 0x8003
  .pword 0x8003
  .pword 0x8003
  .pword 0x8003
  .pword 0x8003
  .pword 0x8003
  .pword 0x8003
  .pword 0x8003
  .pword 0x8003
  .pword 0x8003
  .pword 0x8003
  .pword 0x8003
  .pword 0x8003
  .pword 0x8003
  .pword 0x8003
  .pword 0x8003
  .pword 0x8003
  .pword 0x8003
  .pword 0x8002
  .pword 0x8002
  .pword 0x8002
  .pword 0x8002
  .pword 0x8002
  .pword 0x8002
  .pword 0x8002
  .pword 0x8002
  .pword 0x8002
  .pword 0x8002
  .pword 0x8002
  .pword 0x8002
  .pword 0x8002
  .pword 0x8002
  .pword 0x8002
  .pword 0x8002
  .pword 0x8002
  .pword 0x8002
  .pword 0x8002
  .pword 0x8002
  .pword 0x8002
  .pword 0x8002
  .pword 0x8002
  .pword 0x8002
  .pword 0x8002
  .pword 0x8002
  .pword 0x8002
  .pword 0x8002
  .pword 0x8002
  .pword 0x8002
  .pword 0x8002
  .pword 0x8002
  .pword 0x8002
  .pword 0x8002
  .pword 0x8002
  .pword 0x8002
  .pword 0x8002
  .pword 0x8002
  .pword 0x8002
  .pword 0x8002
  .pword 0x8002
  .pword 0x8002
  .pword 0x8002
  .pword 0x8002
  .pword 0x8002
  .pword 0x8002
  .pword 0x8002
  .pword 0x8002
  .pword 0x8002
  .pword 0x8002
  .pword 0x8002
  .pword 0x8002
  .pword 0x8002
  .pword 0x8002
  .pword 0x8002
  .pword 0x8002
  .pword 0x8002
  .pword 0x8002
  .pword 0x8002
  .pword 0x8002
  .pword 0x8002
  .pword 0x8002
  .pword 0x8002
  .pword 0x8002
  .pword 0x8002
  .pword 0x8002
  .pword 0x8002
  .pword 0x8002
  .pword 0x8002
  .pword 0x8002
  .pword 0x8002
  .pword 0x8002
  .pword 0x8002
  .pword 0x8002
  .pword 0x8002
  .pword 0x8002
  .pword 0x8002
  .pword 0x8002
  .pword 0x8002
  .pword 0x8002
  .pword 0x8002
  .pword 0x8002
  .pword 0x8002
  .pword 0x8002
  .pword 0x8002
  .pword 0x8002
  .pword 0x8002
  .pword 0x8002
  .pword 0x8002
  .pword 0x8002
  .pword 0x8002
  .pword 0x8002
  .pword 0x8002
  .pword 0x8002
  .pword 0x8002
  .pword 0x8002
  .pword 0x8002
  .pword 0x8002
  .pword 0x8002
  .pword 0x8002
  .pword 0x8002
  .pword 0x8002
  .pword 0x8002
  .pword 0x8002
  .pword 0x8002
  .pword 0x8002
  .pword 0x8002
  .pword 0x8002
  .pword 0x8002
  .pword 0x8002
  .pword 0x8002
  .pword 0x8002
  .pword 0x8002
  .pword 0x8002
  .pword 0x8002
  .pword 0x8003
  .pword 0x8003
  .pword 0x8003
  .pword 0x8003
  .pword 0x8003
  .pword 0x8003
  .pword 0x8003
  .pword 0x8003
  .pword 0x8003
  .pword 0x8003
  .pword 0x8003
  .pword 0x8003
  .pword 0x8003
  .pword 0x8003
  .pword 0x8003
  .pword 0x8003
  .pword 0x8003
  .pword 0x8003
  .pword 0x8004
  .pword 0x8004
  .pword 0x8004
  .pword 0x8004
  .pword 0x8004
  .pword 0x8004
  .pword 0x8004
  .pword 0x8004
  .pword 0x8004
  .pword 0x8004
  .pword 0x8004
  .pword 0x8005
  .pword 0x8005
  .pword 0x8005
  .pword 0x8005
  .pword 0x8005
  .pword 0x8005
  .pword 0x8005
  .pword 0x8006
  .pword 0x8006
  .pword 0x8006
  .pword 0x8006
  .pword 0x8006
  .pword 0x8006
  .pword 0x8007
  .pword 0x8007
  .pword 0x8007
  .pword 0x8007
  .pword 0x8007
  .pword 0x8007
  .pword 0x8008
  .pword 0x8008
  .pword 0x8008
  .pword 0x8008
  .pword 0x8009
  .pword 0x8009
  .pword 0x8009
  .pword 0x8009
  .pword 0x800A
  .pword 0x800A
  .pword 0x800A
  .pword 0x800A
  .pword 0x800B
  .pword 0x800B
  .pword 0x800B
  .pword 0x800C
  .pword 0x800C
  .pword 0x800C
  .pword 0x800D
  .pword 0x800D
  .pword 0x800D
  .pword 0x800E
  .pword 0x800E
  .pword 0x800E
  .pword 0x800F
  .pword 0x800F
  .pword 0x8010
  .pword 0x8010
  .pword 0x8011
  .pword 0x8011
  .pword 0x8011
  .pword 0x8012
  .pword 0x8012
  .pword 0x8013
  .pword 0x8013
  .pword 0x8014
  .pword 0x8014
  .pword 0x8015
  .pword 0x8015
  .pword 0x8016
  .pword 0x8016
  .pword 0x8017
  .pword 0x8018
  .pword 0x8018
  .pword 0x8019
  .pword 0x8019
  .pword 0x801A
  .pword 0x801B
  .pword 0x801B
  .pword 0x801C
  .pword 0x801D
  .pword 0x801D
  .pword 0x801E
  .pword 0x801F
  .pword 0x801F
  .pword 0x8020
  .pword 0x8021
  .pword 0x8022
  .pword 0x8022
  .pword 0x8023
  .pword 0x8024
  .pword 0x8025
  .pword 0x8026
  .pword 0x8027
  .pword 0x8027
  .pword 0x8028
  .pword 0x8029
  .pword 0x802A
  .pword 0x802B
  .pword 0x802C
  .pword 0x802D
  .pword 0x802E
  .pword 0x802F
  .pword 0x8030
  .pword 0x8031
  .pword 0x8032
  .pword 0x8033
  .pword 0x8034
  .pword 0x8035
  .pword 0x8036
  .pword 0x8038
  .pword 0x8039
  .pword 0x803A
  .pword 0x803B
  .pword 0x803C
  .pword 0x803E
  .pword 0x803F
  .pword 0x8040
  .pword 0x8041
  .pword 0x8043
  .pword 0x8044
  .pword 0x8045
  .pword 0x8047
  .pword 0x8048
  .pword 0x8049
  .pword 0x804B
  .pword 0x804C
  .pword 0x804E
  .pword 0x804F
  .pword 0x8051
  .pword 0x8052
  .pword 0x8054
  .pword 0x8055
  .pword 0x8057
  .pword 0x8059
  .pword 0x805A
  .pword 0x805C
  .pword 0x805E
  .pword 0x805F
  .pword 0x8061
  .pword 0x8063
  .pword 0x8065
  .pword 0x8067
  .pword 0x8068
  .pword 0x806A
  .pword 0x806C
  .pword 0x806E
  .pword 0x8070
  .pword 0x8072
  .pword 0x8074
  .pword 0x8076
  .pword 0x8078
  .pword 0x807A
  .pword 0x807C
  .pword 0x807E
  .pword 0x8080
  .pword 0x8083
  .pword 0x8085
  .pword 0x8087
  .pword 0x8089
  .pword 0x808C
  .pword 0x808E
  .pword 0x8090
  .pword 0x8093
  .pword 0x8095
  .pword 0x8098
  .pword 0x809A
  .pword 0x809D
  .pword 0x809F
  .pword 0x80A2
  .pword 0x80A4
  .pword 0x80A7
  .pword 0x80A9
  .pword 0x80AC
  .pword 0x80AF
  .pword 0x80B2
  .pword 0x80B4
  .pword 0x80B7
  .pword 0x80BA
  .pword 0x80BD
  .pword 0x80C0
  .pword 0x80C3
  .pword 0x80C6
  .pword 0x80C9
  .pword 0x80CC
  .pword 0x80CF
  .pword 0x80D2
  .pword 0x80D5
  .pword 0x80D8
  .pword 0x80DC
  .pword 0x80DF
  .pword 0x80E2
  .pword 0x80E5
  .pword 0x80E9
  .pword 0x80EC
  .pword 0x80F0
  .pword 0x80F3
  .pword 0x80F7
  .pword 0x80FA
  .pword 0x80FE
  .pword 0x8101
  .pword 0x8105
  .pword 0x8109
  .pword 0x810D
  .pword 0x8110
  .pword 0x8114
  .pword 0x8118
  .pword 0x811C
  .pword 0x8120
  .pword 0x8124
  .pword 0x8128
  .pword 0x812C
  .pword 0x8130
  .pword 0x8134
  .pword 0x8138
  .pword 0x813D
  .pword 0x8141
  .pword 0x8145
  .pword 0x814A
  .pword 0x814E
  .pword 0x8152
  .pword 0x8157
  .pword 0x815B
  .pword 0x8160
  .pword 0x8165
  .pword 0x8169
  .pword 0x816E
  .pword 0x8173
  .pword 0x8178
  .pword 0x817C
  .pword 0x8181
  .pword 0x8186
  .pword 0x818B
  .pword 0x8190
  .pword 0x8195
  .pword 0x819B
  .pword 0x81A0
  .pword 0x81A5
  .pword 0x81AA
  .pword 0x81B0
  .pword 0x81B5
  .pword 0x81BA
  .pword 0x81C0
  .pword 0x81C5
  .pword 0x81CB
  .pword 0x81D0
  .pword 0x81D6
  .pword 0x81DC
  .pword 0x81E2
  .pword 0x81E7
  .pword 0x81ED
  .pword 0x81F3
  .pword 0x81F9
  .pword 0x81FF
  .pword 0x8205
  .pword 0x820C
  .pword 0x8212
  .pword 0x8218
  .pword 0x821E
  .pword 0x8225
  .pword 0x822B
  .pword 0x8231
  .pword 0x8238
  .pword 0x823F
  .pword 0x8245
  .pword 0x824C
  .pword 0x8253
  .pword 0x8259
  .pword 0x8260
  .pword 0x8267
  .pword 0x826E
  .pword 0x8275
  .pword 0x827C
  .pword 0x8283
  .pword 0x828B
  .pword 0x8292
  .pword 0x8299
  .pword 0x82A1
  .pword 0x82A8
  .pword 0x82B0
  .pword 0x82B7
  .pword 0x82BF
  .pword 0x82C6
  .pword 0x82CE
  .pword 0x82D6
  .pword 0x82DE
  .pword 0x82E6
  .pword 0x82EE
  .pword 0x82F6
  .pword 0x82FE
  .pword 0x8306
  .pword 0x830E
  .pword 0x8317
  .pword 0x831F
  .pword 0x8328
  .pword 0x8330
  .pword 0x8339
  .pword 0x8341
  .pword 0x834A
  .pword 0x8353
  .pword 0x835C
  .pword 0x8364
  .pword 0x836D
  .pword 0x8376
  .pword 0x8380
  .pword 0x8389
  .pword 0x8392
  .pword 0x839B
  .pword 0x83A5
  .pword 0x83AE
  .pword 0x83B8
  .pword 0x83C1
  .pword 0x83CB
  .pword 0x83D5
  .pword 0x83DE
  .pword 0x83E8
  .pword 0x83F2
  .pword 0x83FC
  .pword 0x8406
  .pword 0x8410
  .pword 0x841B
  .pword 0x8425
  .pword 0x842F
  .pword 0x843A
  .pword 0x8444
  .pword 0x844F
  .pword 0x845A
  .pword 0x8464
  .pword 0x846F
  .pword 0x847A
  .pword 0x8485
  .pword 0x8490
  .pword 0x849B
  .pword 0x84A6
  .pword 0x84B2
  .pword 0x84BD
  .pword 0x84C8
  .pword 0x84D4
  .pword 0x84E0
  .pword 0x84EB
  .pword 0x84F7
  .pword 0x8503
  .pword 0x850F
  .pword 0x851B
  .pword 0x8527
  .pword 0x8533
  .pword 0x853F
  .pword 0x854B
  .pword 0x8558
  .pword 0x8564
  .pword 0x8571
  .pword 0x857D
  .pword 0x858A
  .pword 0x8597
  .pword 0x85A4
  .pword 0x85B1
  .pword 0x85BE
  .pword 0x85CB
  .pword 0x85D8
  .pword 0x85E6
  .pword 0x85F3
  .pword 0x8600
  .pword 0x860E
  .pword 0x861C
  .pword 0x8629
  .pword 0x8637
  .pword 0x8645
  .pword 0x8653
  .pword 0x8661
  .pword 0x866F
  .pword 0x867E
  .pword 0x868C
  .pword 0x869A
  .pword 0x86A9
  .pword 0x86B7
  .pword 0x86C6
  .pword 0x86D5
  .pword 0x86E4
  .pword 0x86F3
  .pword 0x8702
  .pword 0x8711
  .pword 0x8720
  .pword 0x8730
  .pword 0x873F
  .pword 0x874F
  .pword 0x875E
  .pword 0x876E
  .pword 0x877E
  .pword 0x878E
  .pword 0x879E
  .pword 0x87AE
  .pword 0x87BE
  .pword 0x87CE
  .pword 0x87DE
  .pword 0x87EF
  .pword 0x87FF
  .pword 0x8810
  .pword 0x8821
  .pword 0x8832
  .pword 0x8843
  .pword 0x8854
  .pword 0x8865
  .pword 0x8876
  .pword 0x8887
  .pword 0x8899
  .pword 0x88AA
  .pword 0x88BC
  .pword 0x88CE
  .pword 0x88E0
  .pword 0x88F2
  .pword 0x8904
  .pword 0x8916
  .pword 0x8928
  .pword 0x893A
  .pword 0x894D
  .pword 0x895F
  .pword 0x8972
  .pword 0x8985
  .pword 0x8998
  .pword 0x89AB
  .pword 0x89BE
  .pword 0x89D1
  .pword 0x89E4
  .pword 0x89F7
  .pword 0x8A0B
  .pword 0x8A1E
  .pword 0x8A32
  .pword 0x8A46
  .pword 0x8A5A
  .pword 0x8A6E
  .pword 0x8A82
  .pword 0x8A96
  .pword 0x8AAB
  .pword 0x8ABF
  .pword 0x8AD4
  .pword 0x8AE8
  .pword 0x8AFD
  .pword 0x8B12
  .pword 0x8B27
  .pword 0x8B3C
  .pword 0x8B51
  .pword 0x8B66
  .pword 0x8B7C
  .pword 0x8B91
  .pword 0x8BA7
  .pword 0x8BBD
  .pword 0x8BD3
  .pword 0x8BE9
  .pword 0x8BFF
  .pword 0x8C15
  .pword 0x8C2B
  .pword 0x8C42
  .pword 0x8C58
  .pword 0x8C6F
  .pword 0x8C86
  .pword 0x8C9D
  .pword 0x8CB4
  .pword 0x8CCB
  .pword 0x8CE2
  .pword 0x8CF9
  .pword 0x8D11
  .pword 0x8D28
  .pword 0x8D40
  .pword 0x8D58
  .pword 0x8D70
  .pword 0x8D88
  .pword 0x8DA0
  .pword 0x8DB8
  .pword 0x8DD1
  .pword 0x8DE9
  .pword 0x8E02
  .pword 0x8E1B
  .pword 0x8E34
  .pword 0x8E4D
  .pword 0x8E66
  .pword 0x8E7F
  .pword 0x8E98
  .pword 0x8EB2
  .pword 0x8ECB
  .pword 0x8EE5
  .pword 0x8EFF
  .pword 0x8F19
  .pword 0x8F33
  .pword 0x8F4D
  .pword 0x8F68
  .pword 0x8F82
  .pword 0x8F9D
  .pword 0x8FB7
  .pword 0x8FD2
  .pword 0x8FED
  .pword 0x9008
  .pword 0x9023
  .pword 0x903F
  .pword 0x905A
  .pword 0x9076
  .pword 0x9091
  .pword 0x90AD
  .pword 0x90C9
  .pword 0x90E5
  .pword 0x9102
  .pword 0x911E
  .pword 0x913A
  .pword 0x9157
  .pword 0x9174
  .pword 0x9191
  .pword 0x91AE
  .pword 0x91CB
  .pword 0x91E8
  .pword 0x9205
  .pword 0x9223
  .pword 0x9240
  .pword 0x925E
  .pword 0x927C
  .pword 0x929A
  .pword 0x92B8
  .pword 0x92D7
  .pword 0x92F5
  .pword 0x9314
  .pword 0x9332
  .pword 0x9351
  .pword 0x9370
  .pword 0x938F
  .pword 0x93AE
  .pword 0x93CE
  .pword 0x93ED
  .pword 0x940D
  .pword 0x942D
  .pword 0x944D
  .pword 0x946D
  .pword 0x948D
  .pword 0x94AD
  .pword 0x94CD
  .pword 0x94EE
  .pword 0x950F
  .pword 0x9530
  .pword 0x9551
  .pword 0x9572
  .pword 0x9593
  .pword 0x95B4
  .pword 0x95D6
  .pword 0x95F8
  .pword 0x9619
  .pword 0x963B
  .pword 0x965D
  .pword 0x9680
  .pword 0x96A2
  .pword 0x96C5
  .pword 0x96E7
  .pword 0x970A
  .pword 0x972D
  .pword 0x9750
  .pword 0x9773
  .pword 0x9797
  .pword 0x97BA
  .pword 0x97DE
  .pword 0x9802
  .pword 0x9825
  .pword 0x984A
  .pword 0x986E
  .pword 0x9892
  .pword 0x98B7
  .pword 0x98DB
  .pword 0x9900
  .pword 0x9925
  .pword 0x994A
  .pword 0x996F
  .pword 0x9995
  .pword 0x99BA
  .pword 0x99E0
  .pword 0x9A06
  .pword 0x9A2C
  .pword 0x9A52
  .pword 0x9A78
  .pword 0x9A9E
  .pword 0x9AC5
  .pword 0x9AEC
  .pword 0x9B12
  .pword 0x9B39
  .pword 0x9B60
  .pword 0x9B88
  .pword 0x9BAF
  .pword 0x9BD7
  .pword 0x9BFE
  .pword 0x9C26
  .pword 0x9C4E
  .pword 0x9C76
  .pword 0x9C9F
  .pword 0x9CC7
  .pword 0x9CF0
  .pword 0x9D19
  .pword 0x9D42
  .pword 0x9D6B
  .pword 0x9D94
  .pword 0x9DBD
  .pword 0x9DE7
  .pword 0x9E10
  .pword 0x9E3A
  .pword 0x9E64
  .pword 0x9E8E
  .pword 0x9EB9
  .pword 0x9EE3
  .pword 0x9F0E
  .pword 0x9F38
  .pword 0x9F63
  .pword 0x9F8E
  .pword 0x9FB9
  .pword 0x9FE5
  .pword 0xA010
  .pword 0xA03C
  .pword 0xA068
  .pword 0xA094
  .pword 0xA0C0
  .pword 0xA0EC
  .pword 0xA119
  .pword 0xA145
  .pword 0xA172
  .pword 0xA19F
  .pword 0xA1CC
  .pword 0xA1F9
  .pword 0xA226
  .pword 0xA254
  .pword 0xA282
  .pword 0xA2AF
  .pword 0xA2DD
  .pword 0xA30C
  .pword 0xA33A
  .pword 0xA368
  .pword 0xA397
  .pword 0xA3C6
  .pword 0xA3F5
  .pword 0xA424
  .pword 0xA453
  .pword 0xA483
  .pword 0xA4B2
  .pword 0xA4E2
  .pword 0xA512
  .pword 0xA542
  .pword 0xA572
  .pword 0xA5A2
  .pword 0xA5D3
  .pword 0xA604
  .pword 0xA634
  .pword 0xA666
  .pword 0xA697
  .pword 0xA6C8
  .pword 0xA6FA
  .pword 0xA72B
  .pword 0xA75D
  .pword 0xA78F
  .pword 0xA7C1
  .pword 0xA7F3
  .pword 0xA826
  .pword 0xA859
  .pword 0xA88B
  .pword 0xA8BE
  .pword 0xA8F1
  .pword 0xA925
  .pword 0xA958
  .pword 0xA98C
  .pword 0xA9C0
  .pword 0xA9F4
  .pword 0xAA28
  .pword 0xAA5C
  .pword 0xAA90
  .pword 0xAAC5
  .pword 0xAAFA
  .pword 0xAB2F
  .pword 0xAB64
  .pword 0xAB99
  .pword 0xABCE
  .pword 0xAC04
  .pword 0xAC3A
  .pword 0xAC70
  .pword 0xACA6
  .pword 0xACDC
  .pword 0xAD12
  .pword 0xAD49
  .pword 0xAD80
  .pword 0xADB7
  .pword 0xADEE
  .pword 0xAE25
  .pword 0xAE5C
  .pword 0xAE94
  .pword 0xAECC
  .pword 0xAF04
  .pword 0xAF3C
  .pword 0xAF74
  .pword 0xAFAC
  .pword 0xAFE5
  .pword 0xB01E
  .pword 0xB057
  .pword 0xB090
  .pword 0xB0C9
  .pword 0xB102
  .pword 0xB13C
  .pword 0xB176
  .pword 0xB1B0
  .pword 0xB1EA
  .pword 0xB224
  .pword 0xB25E
  .pword 0xB299
  .pword 0xB2D4
  .pword 0xB30F
  .pword 0xB34A
  .pword 0xB385
  .pword 0xB3C0
  .pword 0xB3FC
  .pword 0xB438
  .pword 0xB474
  .pword 0xB4B0
  .pword 0xB4EC
  .pword 0xB529
  .pword 0xB565
  .pword 0xB5A2
  .pword 0xB5DF
  .pword 0xB61C
  .pword 0xB659
  .pword 0xB697
  .pword 0xB6D5
  .pword 0xB712
  .pword 0xB750
  .pword 0xB78F
  .pword 0xB7CD
  .pword 0xB80B
  .pword 0xB84A
  .pword 0xB889
  .pword 0xB8C8
  .pword 0xB907
  .pword 0xB947
  .pword 0xB986
  .pword 0xB9C6
  .pword 0xBA06
  .pword 0xBA46
  .pword 0xBA86
  .pword 0xBAC6
  .pword 0xBB07
  .pword 0xBB48
  .pword 0xBB89
  .pword 0xBBCA
  .pword 0xBC0B
  .pword 0xBC4C
  .pword 0xBC8E
  .pword 0xBCD0
  .pword 0xBD12
  .pword 0xBD54
  .pword 0xBD96
  .pword 0xBDD9
  .pword 0xBE1B
  .pword 0xBE5E
  .pword 0xBEA1
  .pword 0xBEE4
  .pword 0xBF27
  .pword 0xBF6B
  .pword 0xBFAF
  .pword 0xBFF2
  .pword 0xC036
  .pword 0xC07B
  .pword 0xC0BF
  .pword 0xC103
  .pword 0xC148
  .pword 0xC18D
  .pword 0xC1D2
  .pword 0xC217
  .pword 0xC25D
  .pword 0xC2A2
  .pword 0xC2E8
  .pword 0xC32E
  .pword 0xC374
  .pword 0xC3BA
  .pword 0xC401
  .pword 0xC447
  .pword 0xC48E
  .pword 0xC4D5
  .pword 0xC51C
  .pword 0xC564
  .pword 0xC5AB
  .pword 0xC5F3
  .pword 0xC63B
  .pword 0xC683
  .pword 0xC6CB
  .pword 0xC713
  .pword 0xC75C
  .pword 0xC7A4
  .pword 0xC7ED
  .pword 0xC836
  .pword 0xC87F
  .pword 0xC8C9
  .pword 0xC912
  .pword 0xC95C
  .pword 0xC9A6
  .pword 0xC9F0
  .pword 0xCA3A
  .pword 0xCA85
  .pword 0xCACF
  .pword 0xCB1A
  .pword 0xCB65
  .pword 0xCBB0
  .pword 0xCBFB
  .pword 0xCC47
  .pword 0xCC93
  .pword 0xCCDE
  .pword 0xCD2A
  .pword 0xCD77
  .pword 0xCDC3
  .pword 0xCE0F
  .pword 0xCE5C
  .pword 0xCEA9
  .pword 0xCEF6
  .pword 0xCF43
  .pword 0xCF91
  .pword 0xCFDE
  .pword 0xD02C
  .pword 0xD07A
  .pword 0xD0C8
  .pword 0xD116
  .pword 0xD164
  .pword 0xD1B3
  .pword 0xD202
  .pword 0xD251
  .pword 0xD2A0
  .pword 0xD2EF
  .pword 0xD33F
  .pword 0xD38E
  .pword 0xD3DE
  .pword 0xD42E
  .pword 0xD47E
  .pword 0xD4CE
  .pword 0xD51F
  .pword 0xD56F
  .pword 0xD5C0
  .pword 0xD611
  .pword 0xD662
  .pword 0xD6B4
  .pword 0xD705
  .pword 0xD757
  .pword 0xD7A9
  .pword 0xD7FB
  .pword 0xD84D
  .pword 0xD89F
  .pword 0xD8F2
  .pword 0xD945
  .pword 0xD998
  .pword 0xD9EB
  .pword 0xDA3E
  .pword 0xDA91
  .pword 0xDAE5
  .pword 0xDB39
  .pword 0xDB8D
  .pword 0xDBE1
  .pword 0xDC35
  .pword 0xDC89
  .pword 0xDCDE
  .pword 0xDD33
  .pword 0xDD88
  .pword 0xDDDD
  .pword 0xDE32
  .pword 0xDE87
  .pword 0xDEDD
  .pword 0xDF33
  .pword 0xDF89
  .pword 0xDFDF
  .pword 0xE035
  .pword 0xE08C
  .pword 0xE0E2
  .pword 0xE139
  .pword 0xE190
  .pword 0xE1E7
  .pword 0xE23E
  .pword 0xE296
  .pword 0xE2EE
  .pword 0xE345
  .pword 0xE39D
  .pword 0xE3F6
  .pword 0xE44E
  .pword 0xE4A6
  .pword 0xE4FF
  .pword 0xE558
  .pword 0xE5B1
  .pword 0xE60A
  .pword 0xE663
  .pword 0xE6BD
  .pword 0xE716
  .pword 0xE770
  .pword 0xE7CA
  .pword 0xE824
  .pword 0xE87F
  .pword 0xE8D9
  .pword 0xE934
  .pword 0xE98F
  .pword 0xE9EA
  .pword 0xEA45
  .pword 0xEAA0
  .pword 0xEAFC
  .pword 0xEB57
  .pword 0xEBB3
  .pword 0xEC0F
  .pword 0xEC6B
  .pword 0xECC7
  .pword 0xED24
  .pword 0xED81
  .pword 0xEDDD
  .pword 0xEE3A
  .pword 0xEE97
  .pword 0xEEF5
  .pword 0xEF52
  .pword 0xEFB0
  .pword 0xF00E
  .pword 0xF06C
  .pword 0xF0CA
  .pword 0xF128
  .pword 0xF186
  .pword 0xF1E5
  .pword 0xF244
  .pword 0xF2A3
  .pword 0xF302
  .pword 0xF361
  .pword 0xF3C0
  .pword 0xF420
  .pword 0xF480
  .pword 0xF4DF
  .pword 0xF53F
  .pword 0xF5A0
  .pword 0xF600
  .pword 0xF661
  .pword 0xF6C1
  .pword 0xF722
  .pword 0xF783
  .pword 0xF7E4
  .pword 0xF846
  .pword 0xF8A7
  .pword 0xF909
  .pword 0xF96A
  .pword 0xF9CC
  .pword 0xFA2F
  .pword 0xFA91
  .pword 0xFAF3
  .pword 0xFB56
  .pword 0xFBB9
  .pword 0xFC1B
  .pword 0xFC7F
  .pword 0xFCE2
  .pword 0xFD45
  .pword 0xFDA9
  .pword 0xFE0C
  .pword 0xFE70
  .pword 0xFED4
  .pword 0xFF38
  .pword 0xFF9D
.end
